r"""

 DESCRIPTION:: Compute the quotient of Hyperbolic Space by SL_2 of imaginary quadratic number fields.

 PARAGRAPH DESCRIPTION:: Program for the computation of the quotient of hyperbolic three-space by the Bianchi groups, SL_2 over rings of imaginary quadratic integers.
You can load the program by defining a BianchiOrbifold object using the command 'BianchiOrbifold(c)', where c can be entered either as the discriminant D of the imaginary quadratic number field over which to do the calculation, or a square-free positive integer m, to get D = -4m for m congruent to 1 or 2 mod 4, respectively D = -m for m congruent to 3 mod 4.  

 EXAMPLE:::

 A = BianchiOrbifold(2), B = BianchiOrbifold(7), C = BianchiOrbifold(-11)
 # Now you can use the member instances for computation. 

 AUTHORS:: 

 - Alexander D. Rahm (2010): Bianchi.gp
 - User surface for Bianchi.gp in SAGE: Atin Modi
 - Porting from python2 to python3 by Joel Sjogren

"""

################################################################################################################
# Installation: Unpack the present compressed folder to your hard disk. 
# You can load BianchiGP.sage into online or offline installations of SAGE with the command 
# load("BianchiGP.sage") after having started SAGE from the directory into which you have placed this file. 
################################################################################################################


# -------------------------- Sage code -------------------------------------
#     Copyright (C) 2010 Alexander D. Rahm <Alexander.Rahm@uni.lu>
#
# Bianchi.gp is a free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, Version 2.1.3 of July 21, 2012. 
#
#---------------------------------------------------------------------------

from sage.interfaces.gp import gp
from sage.arith.misc import is_squarefree
from sage.rings.number_field.number_field import QuadraticField
from sage.misc.sage_eval import sage_eval
from sage.matrix.constructor import matrix
from sage.functions.other import ceil, sqrt

gp(r'read("Bianchi.gp");')
gp(r'read("Bianchi-sage-compat.gp");')

print("BianchiGP loaded into SAGE. Type \"BianchiOrbifold?\" for help (taking care not to indent this command).")

class BianchiOrbifold:
  """ 
  Returns a Bianchi Orbifold Object for a given discriminant D of an imaginary quadratic number field. This Bianchi Orbifold Object contains various properties 
  of the Bianchi Fundamental Polyhedron like cell structure, coordinates of the vertices, stabilizers, identification matrices and many more. With the member function    group_homology(), the homology of the Bianchi group can be calculated and displayed on the screen. 

  INPUT:

  - You can either enter the discriminant D of the imaginary quadratic number field over which to do the calculation, or you can enter a square-free positive integer m, to get D = -4m for m congruent to 1 or 2 mod 4, respectively D = -m for m congruent to 3 mod 4. 


  OUTPUT: Bianchi Orbifold Object

  Type 

  BianchiOrbifold.help? 

  (taking care not to indent this command) to see a list of the properties of the Bianchi orbifold that are recorded into the Bianchi Orbifold Object, and how to access them.

  EXAMPLES::

    sage:A = BianchiOrbifold(2) #Defining the Bianchi Orbifold object by choosing a positive square-free integer.
    sage:B = BianchiOrbifold(-11) #Defining the Bianchi Orbifold object using the discriminant value for the field.

  .. NOTE::

    Higher discriminant absolute values |D| will require more computation time.

  """
  def __init__(self,c):
    """
    Constructs a Bianchi orbifold object.

    If c is negative, it is treated as a discriminant D.
    D must be either 0 mod 4 and -D/4 1 or 2 mod 4, or -3 mod 4,
    and furthermore it must be fundamental.

    If c is positive, it is treated as m, with w^2 + m = 0.
    m must be square-free and distinct from 1 and 3, which are not supported.
    """
    if c < 0:
      m = BianchiOrbifold.discriminant_to_m_converter(c)
      if m:
        self.m = m
      else:
        raise ValueError("invalid discriminant {}".format(c))
    elif c == 0:
      raise ValueError("m = {} must be nonzero".format(c))
    elif c in {1,3}:
      raise ValueError("computations for m = 1 and m = 3 are incomplete")
    elif not is_squarefree(c):
      raise ValueError("m must be square-free")
    else:
      self.m = c
      
    self.w = QuadraticField(-self.m, 'w').gen()

  def _latex_(self):
  
    return r'\mathbb{H}^3/\mathrm{PSL}_2(\mathcal{O}_%s)'%(self.m)

  def __repr__(self):
    
    return "Bianchi Orbifold Object for %s."%(self.w.parent())

  @staticmethod
  def discriminant_to_m_converter(D):
    access_gp = lambda gpname: sage_eval(gp.get(gpname))
    fund = access_gp('isfundamental({})'.format(D))

    if not fund == 1:
      return None
    if D%4 == 0 and (-D//4)%4 in {1,2}:
      return -D//4
    if -D%4 == 3:
      return -D

  def help(self):
    """   
    To access any member function, define a Bianchi Orbifold Object and then use the dot operator. 
    To get help for that function, put a question mark after typing the 
    function name. 

    EXAMPLES::

    sage:A = BianchiOrbifold(2)

    # To access the member function number_of_lines() and its documentation:

    sage: A.number_of_lines()

    16 

    # To access the documentation for the function number_of_lines()

    sage:A.number_of_lines?
    
    LIST OF FUNCTIONS::

    number_of_lines(): Returns number of lines along which two hemispheres intersect. On these lines lie the edges of the Bianchi Fundamental Polyhedron.

    max_neighbours_number(): Returns the maximum number of neighbours of a hemisphere amongst all hemispheres in the list of hemispheres centered in the fundamental rectangle that are closer to the cusp at infinity than (or as close to it as) any other hemisphere.

    total_point_set(): Returns the set of points where two intersection lines of hemispheres intersect, and the point is not below any other hemisphere. 
    The vertices of the Bianchi Fundamental Polyhedron are these intersection points. 

    points_of_sphere(): Returns a list of coordinates of points in the hyperbolic plane for each hemisphere in the hemisphere list which are either vertices or corner points. 

    points_of_line(): Returns the end points of every line defined through intersection of two hemispheres. 

    edges_list(): Returns the end points of all edges of the convex polygon defined for the Bianchi Fundamental Polyhedron. These are obtained after deleting some 
    lines which do not satisfy Swan's Criterion. 

    two_cell_support(): Returns the list of indices which for each 2-cell of the Bianchi Fundamental Polyhedron points to the hemisphere on which the 2-cell lies.

    sphere_center(): Returns the coordinates of centers of the hemispheres required to define the Bianchi Fundamental Polyhedron. 

    radius_square(): Returns the square of the radius of each hemisphere in the hemispheres list.

    neighbours(): Returns the index numbers of all neighbours of a hemisphere. 

    lines_of_sphere(): Returns a list of lines for each hemisphere required to define the Bianchi Fundamental Polyhedron. Each line is represented through a point on the line and a direction vector. 

    stabilizer(): Returns a list of stabilizer groups (represented as lists of matrices) in the Bianchi group which fix a vertex of the Bianchi Fundamental Polyhedron.
     
    edge_stabilizer(): Returns a list of stabilizer groups (represented as lists of matrices) in the Bianchi group which fix an edge of the Bianchi Fundamental Polyhedron.


    identification_matrices(): Returns the list of matrices of the Bianchi group which identifies two points in the list of vertices of the convex polygon as 
    defined for the Bianchi fundamental Polyhedron.  

    equivalent_vertices(): Returns the list of indices of vertices which are identified by a matrix of the Bianchi group.

    transport_from(): Returns a list of indices of matrices. The indices in this list are according to the index in the function identification_matrices.

    equivalent_edges(): Returns the list of indices of edges which are identified or whose end points are in the same orbit.

    edges_of_2cell(): Returns the list of indices of edges which forms a 2-cell in the Bianchi fundamental polyhedron indexed according to the hemispheres which 
    forms a 2-cell listed in the function two_cell_support.

    edge_origin(): Returns a list of indices of vertices through which an edge of the convex polygon defined for Bianchi Fundamental Polyhedron originates. It helps 
    in providing a orientation for the edge by combining with the list generated through the function edge_end.

    edge_end(): Returns a list of indices of vertices through which an edge of the convex polygon defined for Bianchi Fundamental Polyhedron ends. It helps in 
    providing a orientation for the edge by combining with the list generated through the function edge_origin.

    vertex_orbit_number(): Returns the orbit number or label for the set of vertices.

    vertex_orbit_representative(): Returns a list of indices of vertices in which each index represents a vertex contained in that indexed orbit.

    number_of_vertex_orbits(): Returns the number of partitions of the vertex set done through identifying vertices using matrices from the Bianchi group.

    number_of_edge_orbits(): Returns the number of partitions of the edge set done through identifying vertices using matrices from the Bianchi group.

    edge_orbit_number(): Returns the orbit number or label for the set of edges.

    edge_orbit_representative(): Returns a list of indices of edges in which each index represents an edge contained in that indexed orbit.

    mu(): Returns the changing value of Mu, an element in the ring of integers of the imaginary quadratic field, till it reaches an expected value, starting with 
    smallest value which the norm of a non-zero element in the ring of integers can take i.e. 1. The radius of the hemispheres required to define Bianchi Fundamental 
    Polyhedron is 1/Mu.

    Lambda(): Returns the changing value of Lambda, an element in the ring of integers of the imaginary quadratic field, till it reaches an expected value. It is 
    changed according to the change in Mu so that the center of hemisphere which Lambda/Mu remains in the fundamental rectangle.

    number_of_spheres(): Returns the number of hemispheres which are required to define the Bianchi Fundamental Polyhedron.

    corners_of_2cell(): Returns the vertices of the 2-cells defined in the cell structure for the boundary of the Bianchi Fundamental Polyhedron.

    delete_cell_flag(): Returns whether a particular 2-cell originally defined for a hemisphere has been deleted or not when the quotient space is formed.

    boundary_matrix(): Returns a matrix which tells for a particular edge in an edge orbit, the information about the vertex orbit number of the origin and end 
    points of the edge.

    number_of_2cells(): Returns the number of 2-cells present in the cell structure of the boundary of the Bianchi Fundamental Polyhedron.

    cell_boundary_matrix(): Returns a matrix which tells for a particular 2-cell in the boundary of the Bianchi Fundamental Polyhedron, the information about the edge 
    orbit number and the orientation of the edge.

    class_number(): Returns the class number for the ring of integers defined for the imaginary quadratic field for a particular value of m.

    edge_two_dim(): Returns the number of elements which contributes to the 2-primary part of E^1_{1,1}.

    edge_2_label(): Returns the edge orbit numbers which contributes to the 2-primary part of E^1_{1,1}.

    edge_three_dim(): Returns the number of elements which contributes to the 3-primary part of E^1_{1,1}.

    edge_3_label(): Returns the edge orbit numbers which contributes to the 3-primary part of E^1_{1,1}.

    group_homology(): Returns the group group homology for the Bianchi group defined for a particular imaginary quadratic field.

    bianchi_calc(): Returns all the necessary properties of the Bianchi Orbifold. You can access individual properties by using various member functions. Type BianchiOrbifold.help? to see the list of properties this program supports.

    """

  def bianchi_calc(self):
    """
    Returns all the necessary properties of the Bianchi Orbifold. You can access individual properties by using various member functions. Type BianchiOrbifold.help?
    to see the list of properties this program supports.

    """
    print(self._access('bianchi_calc'))

  def _access(self, attr):
    self._force()
    return self._everything[attr]

  def _allocate_memory(self):
    try:
        import psutil
        available_at_most = (psutil.virtual_memory().total + psutil.swap_memory().total) // (2**20)
    except:
        print("Warning: unable to estimate available memory (psutil required)")
        available_at_most = None

    error = lambda req: MemoryError(
      "There is not enough memory for the BianchiOrbifold computation ({} MB vs {} MB).".format(
        available_at_most, req))

    known_limit = BianchiOrbifold._known_limit(self.m)
        
    if known_limit:
      if available_at_most and available_at_most < known_limit:
        raise error(known_limit)
      else:
        estimate_megabytes = known_limit
    else:
      if available_at_most and available_at_most < 60*self.m:
        raise error(60*self.m)
      else:
        estimate_megabytes = BianchiOrbifold._estimate_limit(self.m)

    gp.get('allocatemem({}*2^20)'.format(estimate_megabytes))

  def _force(self):
    """
    Runs the whole computation and stores all results for later access by the user.
    If the computation has already been performed, does nothing.
    """
    if not hasattr(self, '_everything'):
      self._allocate_memory()

      output = gp.get('Start('+str(self.m)+')')
      output = output.replace('\nTo save the computed data in backup files in the current folder, type the command BackUp()\nTo output a selection of the obtained information in .tex files, type the command OutputFiles()\n0','')

      if '***' in output:
        raise ChildProcessError(
          "An error occurred in the gp implementation of BianchiOrbifold:\n\n{}".format(output))

      access_gp = lambda gpname: sage_eval(gp.get(gpname),locals={'w':self.w})

      self._everything = {
        'bianchi_calc': output,
        'group_homology': gp.get('ComputeGroupHomology()'),
        'number_of_lines': access_gp('numberOfLines'),
        'max_neighbours_number': access_gp('maxNeighboursNumber'),
        'total_point_set': access_gp('totalPointSetsageConverter()'),
        'points_of_sphere': access_gp('pointsOfSpheresageConverter()'),
        'points_of_line': access_gp('pointsOfLinesageConverter()'),
        'edges_list': access_gp('edgesListsageConverter()'),
        'two_cell_support': access_gp('Vec(twoCellSupport)'),
        'sphere_center': access_gp('sphereCentersageConverter()'),
        'radius_square': access_gp('radiusSquare'),
        'neighbours': access_gp('neighbours'),
        'lines_of_sphere': access_gp('linesOfSpheresageConverter()'),
        'stabilizer': BianchiOrbifold._gptosagematrixConverter(
          access_gp('gpmatrixtosagematrixConverter(stabilizer)')),
        'edge_stabilizer': BianchiOrbifold._gptosagematrixConverter(
          access_gp('gpmatrixtosagematrixConverter(edgeStabilizer)')),
        'identification_matrices': BianchiOrbifold._gptosagematrixConverter(
          access_gp('Vec(gpmatrixtosagematrixConverter(identificationMatrices))')),
        'equivalent_vertices': access_gp('listtovectorsageconverter(equivalentVertices)'),
        'transport_from': access_gp('listtovectorsageconverter(transportFrom)'),
        'equivalent_edges': access_gp('listtovectorsageconverter(equivalentEdges)'),
        'edges_of_2cell': access_gp('listtovectorsageconverter(edgesOf2cell)'),
        'edge_origin': access_gp('EdgeOrigin'),
        'edge_end': access_gp('EdgeEnd'),
        'vertex_orbit_number': access_gp('vertexOrbitNumber'),
        'vertex_orbit_representative': access_gp('vertexOrbitRepresentative'),
        'number_of_vertex_orbits': access_gp('numberOfVertexOrbits'),
        'number_of_edge_orbits': access_gp('numberOfEdgeOrbits'),
        'edge_orbit_number': access_gp('edgeOrbitNumber'),
        'edge_orbit_representative': access_gp('edgeOrbitRepresentative'),
        'mu': access_gp('MusageConverter(Mu)'),
        'Lambda': access_gp('MusageConverter(Lambda)'),
        'number_of_spheres': access_gp('numberOfSpheres'),
        'corners_of_2cell': access_gp('cornersOf2cell'),
        'delete_cell_flag': access_gp('Vec(deleteCellFlag)'),
        'boundary_matrix': BianchiOrbifold._gptosagerectmatrixConverter(
          access_gp('gprectMatrixsageConverter(boundaryMatrix)')),
        'number_of_2cells': access_gp('numberOfTwoCells'),
        'cell_boundary_matrix': BianchiOrbifold._gptosagerectmatrixConverter(
          access_gp('gprectMatrixsageConverter(CellBoundaryMatrix)')),
        'class_number': access_gp('classNumber'),
        'edge_two_dim': access_gp('EdgeTwoDim'),
        'edge_2_label': access_gp('Vec(edge2Label)'),
        'edge_three_dim': access_gp('EdgeThreeDim'),
        'edge_3_label': access_gp('Vec(edge3Label)')
      }

  def group_homology(self):
    """
    Returns the group group homology for the Bianchi group defined for a particular imaginary quadratic field.

    """

    print(self._access('group_homology'))

  def number_of_lines(self):
    """
    Returns number of lines along which two hemispheres intersect. On these lines lie the edges of the Bianchi Fundamental Polyhedron.

    EXAMPLE:::

      sage:A = BianchiOrbifold(2) 
      sage:A.number_of_lines()
      16

    TESTS:::

      sage:A = BianchiOrbifold(5) # Check for 1mod4case
      sage:A.number_of_lines()
      44
      sage:B = BianchiOrbifold(2) # Check for 2mod4 case
      sage:B.number_of_lines()
      16
      sage:C = BianchiOrbifold(7) # Check for 3mod4 case
      sage:C.number_of_lines()
      9
    """

    return self._access('number_of_lines')

  def max_neighbours_number(self):
    """
    Returns the maximum number of neighbours of a hemisphere amongst all hemispheres in the list of hemispheres centered in the fundamental rectangle that are closer to the cusp at infinity than (or as close to it as) any other hemisphere.

    EXAMPLE::: 

      sage:A = BianchiOrbifold(2)
      sage:A.max_neighbours_number()
      sage:3

    .. NOTE::

      The hemisphere list consist of all necessary hemispheres in the rectangle which are necessary to define the Bianchi Fundamental Polyhedron. A hemisphere
      is a neighbour of other hemisphere if they touch at some point in the hyperbolic plane.

    TESTS:::

      sage:A = BianchiOrbifold(5) # Check for 1mod4case
      sage:A.number_of_lines()
      7
      sage:B = BianchiOrbifold(2) # Check for 2mod4 case
      sage:B.number_of_lines()
      3
      sage:C = BianchiOrbifold(7) # Check for 3mod4 case
      sage:C.number_of_lines()
      2
    """

    return self._access('max_neighbours_number')

  def total_point_set(self):
    """
    Returns the set of points where two intersection lines of hemispheres intersect, and the point is not below any other hemisphere. 
    The vertices of the Bianchi Fundamental Polyhedron are these intersection points. The imaginary part of the coordinates in the complex plane is represented using the imaginary quadratic base element `w` in the imaginary quadratic number field. This base element is saved as `B.w` if `B` is our Bianchi orbifold object.

    EXAMPLE:: 

      sage:A = BianchiOrbifold(2)
      sage:A.total_point_set()
      [[0, 1],
      [w, 1],
      [1/2*w, 1/2],
      [1, 1],
      [w + 1, 1],
      [1/2*w + 1, 1/2],
      [1/2, 3/4],
      [w + 1/2, 3/4],
      [1/2*w + 1/2, 1/4]]

    .. NOTE::

      The points obtained are coordinates in the upper half-space model for hyperbolic three-space.  For example in the point represented as [1/2*w + 1, 1/2] admits the projection `1/2*w + 1` to the complex plane and `1/2` as its height in hyperbolic three-space.

    TESTS::

      sage:A = BianchiOrbifold(5) # Check for 1mod4case
      sage:A.total_point_set()
      [[0, 1],
       [w, 1],
       [2/5*w, 1/5],
       [3/5*w, 1/5],
       [1, 1],
       [w + 1, 1],
       [2/5*w + 1, 1/5],
       [3/5*w + 1, 1/5],
       [1/2, 3/4],
       [w + 1/2, 3/4],
       [1/2*w + 1/2, 0],
       [3/8*w + 1/2, 3/64],
       [5/8*w + 1/2, 3/64],
       [2/5*w + 2/5, 1/25],
       [3/5*w + 2/5, 1/25],
       [2/5*w + 3/5, 1/25],
       [3/5*w + 3/5, 1/25]]
      sage: B = BianchiOrbifold(2) # Check for 2mod4 case
      sage: B.total_point_set()
      [[0, 1],
       [w, 1],
       [1/2*w, 1/2],
       [1, 1],
       [w + 1, 1],
       [1/2*w + 1, 1/2],
       [1/2, 3/4],
       [w + 1/2, 3/4],
       [1/2*w + 1/2, 1/4]]
      sage: C = BianchiOrbifold(7) # Check for 3mod4 case
      sage: C.total_point_set()
      [[0, 1],
       [w, 1],
       [w + 1, 1],
       [3/7*w - 2/7, 3/7],
       [-1/2, 3/4],
       [1/2, 3/4],
       [w + 1/2, 3/4],
       [4/7*w + 2/7, 3/7],
       [3/7*w + 5/7, 3/7]]
    """

    return self._access('total_point_set')

  def points_of_sphere(self):
    """
    Returns a list of coordinates of points in the hyperbolic plane for each hemisphere in the hemisphere list which are either vertices or corner points. The imaginary part of the coordinates in the complex plane is represented using the imaginary quadratic base element `w` in the imaginary quadratic number field. This base element is saved as `B.w` if `B` is our Bianchi orbifold object.

    EXAMPLE:: 

      sage:A = BianchiOrbifold(2)
      sage:A.points_of_sphere()
      [[[0, 1], [1/2*w, 1/2], [1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[w, 1], [1/2*w, 1/2], [w + 1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[1, 1], [1/2*w + 1, 1/2], [1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[w + 1, 1], [1/2*w + 1, 1/2], [w + 1/2, 3/4], [1/2*w + 1/2, 1/4]]]

    .. NOTE::
  
      This function prints a list of points corresponding to each hemisphere which are either vertices or corner points. You can use indexing in matrices to get the list of coordinates for a specific sphere. The points obtained are coordinates in hyperbolic three-space. For example in the point represented as [1/2*w + 1, 1/2] admits the projection `1/2*w + 1` to the complex plane and `1/2` as its height in hyperbolic three-space.

    .. SEEALSO::

      :function: `number_of_spheres`
      :function: `total_point_set`

    TESTS::

      sage:A = BianchiOrbifold(5) # Check for 1mod4case
      sage:A.points_of_sphere()
      [[[0, 1], [2/5*w, 1/5], [1/2, 3/4], [3/8*w + 1/2, 3/64], [2/5*w + 2/5, 1/25]],
       [[w, 1],
        [3/5*w, 1/5],
        [w + 1/2, 3/4],
        [5/8*w + 1/2, 3/64],
        [3/5*w + 2/5, 1/25]],
       [[1, 1],
        [2/5*w + 1, 1/5],
        [1/2, 3/4],
        [3/8*w + 1/2, 3/64],
        [2/5*w + 3/5, 1/25]],
       [[w + 1, 1],
        [3/5*w + 1, 1/5],
        [w + 1/2, 3/4],
        [5/8*w + 1/2, 3/64],
        [3/5*w + 3/5, 1/25]],
       [[2/5*w, 1/5],
        [3/5*w, 1/5],
        [1/2*w + 1/2, 0],
        [2/5*w + 2/5, 1/25],
        [3/5*w + 2/5, 1/25]],
       [[2/5*w + 1, 1/5],
        [3/5*w + 1, 1/5],
        [1/2*w + 1/2, 0],
        [2/5*w + 3/5, 1/25],
        [3/5*w + 3/5, 1/25]],
       [[1/2*w + 1/2, 0],
        [5/8*w + 1/2, 3/64],
        [3/5*w + 2/5, 1/25],
        [3/5*w + 3/5, 1/25]],
       [[1/2*w + 1/2, 0],
        [3/8*w + 1/2, 3/64],
        [2/5*w + 2/5, 1/25],
        [2/5*w + 3/5, 1/25]]]
      sage: B = BianchiOrbifold(2) # Check for 2mod4 case
      sage: B.points_of_sphere()
      [[[0, 1], [1/2*w, 1/2], [1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[w, 1], [1/2*w, 1/2], [w + 1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[1, 1], [1/2*w + 1, 1/2], [1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[w + 1, 1], [1/2*w + 1, 1/2], [w + 1/2, 3/4], [1/2*w + 1/2, 1/4]]]
      sage: C = BianchiOrbifold(7) # Check for 3mod4 case
      sage: C.points_of_sphere()
      [[0, 1],
       [w, 1],
       [w + 1, 1],
       [3/7*w - 2/7, 3/7],
       [-1/2, 3/4],
       [1/2, 3/4],
       [w + 1/2, 3/4],
       [4/7*w + 2/7, 3/7],
       [3/7*w + 5/7, 3/7]]
    """
    
    return self._access('points_of_sphere')

  def points_of_line(self):
    """

    Returns the end points of every line defined through intersection of two hemispheres. The imaginary part of the coordinates in the complex plane is represented using the imaginary quadratic base element `w` in the imaginary quadratic number field. This base element is saved as `B.w` if `B` is our Bianchi orbifold object.

    EXAMPLE:: 

      sage:A = BianchiOrbifold(2)
      sage:A.points_of_line()
      [[[[1/2*w, 1/2], [1/2*w + 1/2, 1/4]],
       [[1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[0, 1], [1/2*w, 1/2]],
       [[0, 1], [1/2, 3/4]]],
       [[[1/2*w, 1/2], [1/2*w + 1/2, 1/4]],
       [[w + 1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[w, 1], [1/2*w, 1/2]],
       [[w, 1], [w + 1/2, 3/4]]],
      [[[1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[1/2*w + 1, 1/2], [1/2*w + 1/2, 1/4]],
       [[1, 1], [1/2*w + 1, 1/2]],
       [[1, 1], [1/2, 3/4]]],
       [[[w + 1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[1/2*w + 1, 1/2], [1/2*w + 1/2, 1/4]],
       [[w + 1, 1], [1/2*w + 1, 1/2]],
       [[w + 1, 1], [w + 1/2, 3/4]]]]

    .. NOTE::
  
      This function prints a list of end points corresponding to each line. You can use indexing in matrices to get the list of end points coordinates for a specific line. The points obtained are coordinates in hyperbolic three-space. For example in the point represented as [1/2*w + 1, 1/2] admits the projection `1/2*w + 1` to the complex plane and `1/2` as its height in hyperbolic three-space.

    .. SEEALSO::

      :function: `number_of_lines`

    """
    
    return self._access('points_of_line')

  def edges_list(self):
    """ 
    Returns the end points of all edges of the convex polygon defined for the Bianchi Fundamental Polyhedron. These are obtained after deleting some 
    lines which do not satisfy Swan's Criterion. The imaginary part of the coordinates in the complex plane is represented using the imaginary quadratic base element `w` in the imaginary quadratic number field. This base element is saved as `B.w` if `B` is our Bianchi orbifold object.

    EXAMPLE:: 

      sage:A = BianchiOrbifold(2)
      sage:A.egdes_list()
      [[[0, 1], [1/2*w, 1/2]],
       [[0, 1], [1/2, 3/4]],
       [[w, 1], [1/2*w, 1/2]],
       [[w, 1], [w + 1/2, 3/4]],
       [[1/2*w, 1/2], [1/2*w + 1/2, 1/4]],
       [[1, 1], [1/2*w + 1, 1/2]],
       [[1, 1], [1/2, 3/4]],
       [[w + 1, 1], [1/2*w + 1, 1/2]],
       [[w + 1, 1], [w + 1/2, 3/4]],
       [[1/2*w + 1, 1/2], [1/2*w + 1/2, 1/4]],
       [[1/2, 3/4], [1/2*w + 1/2, 1/4]],
       [[w + 1/2, 3/4], [1/2*w + 1/2, 1/4]]]

    .. NOTE::
  
      This function prints a list of end points corresponding to each edge. You can use indexing in matrices to get the list of end points coordinates for a specific edge. The points obtained are coordinates in hyperbolic three-space. For example in the point represented as [1/2*w + 1, 1/2] admits the projection `1/2*w + 1` to the complex plane and `1/2` as its height in hyperbolic three-space.

    .. SEEALSO::

      :function: `points_of_lines`

    """   

    return self._access('edges_list')

  def two_cell_support(self):
    """
    Returns the list of indices which for each 2-cell of the Bianchi Fundamental Polyhedron points to the hemisphere on which the 2-cell lies.

    EXAMPLE:: 

      sage:A = BianchiOrbifold(2)
      sage:A.two_cell_support(2)
      returns: "Hemisphere centered at ", sphereCenter[two_cell_support[2]]," of radius square ",radiusSquare[two_cell_support[2]]
      [1, 2]

    .. NOTE::
  
      This function prints a list of indices of sphere according to the order as provided in the function sphere_center.

    .. SEEALSO::

      :function: `sphere_center`

    """   

    return self._access('two_cell_support')

  def sphere_center(self):
    """ 
    Returns the coordinates of centers of the hemispheres required to define the Bianchi Fundamental Polyhedron. The imaginary part of the coordinates in the complex plane is represented using the imaginary quadratic base element `w` in the imaginary quadratic number field. This base element is saved as `B.w` if `B` is our Bianchi orbifold object.


    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.sphere_center()
      [0, w, 1, w + 1]

    .. NOTE::
  
      This function prints a list of coordinates of centers corresponding to each sphere. You can use indexing in matrices to get the coordinates of center
      for a specific sphere. The points obtained are coordinates in the complex plane. 

    .. SEEALSO::

      :function: `number_of_spheres`

    """   

    return self._access('sphere_center')

  def radius_square(self):
    """ 
    Returns the square of the radius of each hemisphere in the hemispheres list.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.radius_square()  
      [1, 1, 1, 1]

    .. NOTE::
  
      This function prints square of the radius corresponding to each sphere. You can use indexing in matrices to get the radius for a specific sphere.

    .. SEEALSO::

      :function: `number_of_spheres`
      :function: `sphere_center`

    """   

    return self._access('radius_square')

  def neighbours(self):
    """ 
    Returns the index numbers of all neighbours of a hemisphere. 

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.neighbours() 
      [[2, 3, 4], [1, 3, 4], [1, 2, 4], [1, 2, 3]]

    .. NOTE::
  
      A hemisphere is a neighbour of other hemisphere if they touch at some point in the hyperbolic plane. This indexing is defined according to the spheres in the list of the function sphere_center. You can use indexing in matrices to get the list of neighbouring
      hemispheres for a specific hemisphere.

    .. SEEALSO::

      :function: `number_of_spheres`
      :function: `sphere_center`

    """   

    return self._access('neighbours')

  def lines_of_sphere(self):
    """ 
    Returns a list of lines for each hemisphere required to define the Bianchi Fundamental Polyhedron. Each line is represented through a point on the line and a
direction vector.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.lines_of_sphere()  
      [[[1/2*w, 1], [1/2, w], [0, w], [0, 1/2]],
       [[1/2*w, 1], [1/2, w], [0, w], [w, 1/2]],
       [[1/2, w], [1/2*w, 1], [w + 1, -1/2*w], [1, -1/2]],
       [[1/2, w], [1/2*w, 1], [w + 1, -1/2*w], [w + 1, -1/2]]]

    .. NOTE::
  
      This indexing is defined according to the hemispheres order in the list of the function sphere_center. You can use indexing in matrices to get the list of lines for a specific hemisphere. These lines are represented in the projected complex plane. For example in the set [w + 1, -1/2*w] `w + 1` is a point on the line in complex plane and `-1/2*w` is the direction vector of the line.

    .. SEEALSO::

      :function: `number_of_lines`
      :function: `number_of_spheres`
      :function: `points_of_line`
    """   

    return self._access('lines_of_sphere')

  def stabilizer(self):
    """
    Returns a list of stabilizer groups (represented as lists of matrices) in the Bianchi group which fix a vertex of the Bianchi Fundamental Polyhedron. The order of the list is the one produced by the member function total_point_set(), which allows to identify the corresponding vertex. The imaginary entries in the matrices are represented using the imaginary quadratic base element `w` in the imaginary quadratic number field. This base element is saved as `B.w` if `B` is our Bianchi orbifold object.


    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.stabilizer() 
      [[
      [1 0]  [-1  0]  [ 0  1]  [ 0 -1]
      [0 1], [ 0 -1], [-1  0], [ 1  0]
      ],
       [
      [1 0]  [-1  0]  [ w -1]  [-w  1]
      [0 1], [ 0 -1], [-1 -w], [ 1  w]
      ],
       [
      [1 0]  [-1  0]  [-1 -w]  [ 1  w]  [ w -1]  [-w  1]  [ 0  1]  [ 0 -1]
      [0 1], [ 0 -1], [-w  1], [ w -1], [-1 -w], [ 1  w], [-1  0], [ 1  0]
      ],
       [
      [1 0]  [-1  0]  [ 1  2]  [-1 -2]
      [0 1], [ 0 -1], [-1 -1], [ 1  1]
      ],
       [
      [1 0]  [-1  0]  [ w + 1    2*w]  [-w - 1   -2*w]
      [0 1], [ 0 -1], [    -1 -w - 1], [     1  w + 1]
      ],
       [
      [1 0]  [-1  0]  [ w - 1     -2]  [ w + 1    2*w]  [ 1  2]  [-1 -2]
      [0 1], [ 0 -1], [    -w -w + 1], [    -1 -w - 1], [-1 -1], [ 1  1],

      [-w - 1   -2*w]  [-w + 1      2]
      [     1  w + 1], [     w  w - 1]
      ],
       [
      [1 0]  [-1  0]  [ 1  1]  [-1 -1]  [ 0  1]  [ 0 -1]
      [0 1], [ 0 -1], [-1  0], [ 1  0], [-1 -1], [ 1  1]
      ],
       [
      [1 0]  [-1  0]  [w + 1 w - 1]  [     w  w - 1]  [    -w -w + 1]
      [0 1], [ 0 -1], [   -1    -w], [    -1 -w - 1], [     1  w + 1],

      [-w - 1 -w + 1]
      [     1      w]
      ],
       [
      [1 0]  [-1  0]  [ w - 1     -2]  [-w + 1      2]  [-1 -w]  [ 1  w]
      [0 1], [ 0 -1], [    -w -w + 1], [     w  w - 1], [-w  1], [ w -1],

      [ w + 1      w]  [-w - 1     -w]  [     w     -1]  [   -w     1]
      [    -2 -w - 1], [     2  w + 1], [-w - 1 -w + 1], [w + 1 w - 1],

      [ 0 -1]  [ 0  1]  [    -w -w + 1]  [     w  w - 1]  [    2 w + 1]
      [ 1  1], [-1 -1], [     1  w + 1], [    -1 -w - 1], [w - 1    -1],

      [    -2 -w - 1]  [ w - 1     -1]  [-w + 1      1]  [    -1 -w - 1]
      [-w + 1      1], [-w - 1     -w], [ w + 1      w], [-w + 1      2],

      [    1 w + 1]  [w + 1 w - 1]  [-w - 1 -w + 1]  [ 1  1]  [-1 -1]
      [w - 1    -2], [   -1    -w], [     1      w], [-1  0], [ 1  0]
      ]]

    .. NOTE::
  
      The order of the list is the one produced by the member function total_point_set(), which allows to identify the corresponding vertex.

    .. SEEALSO::

      :function: `total_point_set`

    """ 

    return self._access('stabilizer')

  def edge_stabilizer(self):
    """
    Returns a list of stabilizer groups (represented as lists of matrices) in the Bianchi group which fix an edge of the Bianchi Fundamental Polyhedron. The imaginary entries in the matrices are represented using the imaginary quadratic base element `w` in the imaginary quadratic number field. This base element is saved as `B.w` if `B` is our Bianchi orbifold object.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.egde_stabilizer()
      [[
      [-1  0]  [1 0]  [ 0  1]  [ 0 -1]
      [ 0 -1], [0 1], [-1  0], [ 1  0]
      ],
       [
      [-1  0]  [1 0]
      [ 0 -1], [0 1]
      ],
       [
      [-1  0]  [1 0]  [-w  1]  [ w -1]
      [ 0 -1], [0 1], [ 1  w], [-1 -w]
      ],
       [
      [-1  0]  [1 0]
      [ 0 -1], [0 1]
      ],
       [
      [-1  0]  [1 0]  [-1 -w]  [ 1  w]
      [ 0 -1], [0 1], [-w  1], [ w -1]
      ],
       [
      [-1  0]  [1 0]  [-1 -2]  [ 1  2]
      [ 0 -1], [0 1], [ 1  1], [-1 -1]
      ],
       [
      [-1  0]  [1 0]
      [ 0 -1], [0 1]
      ],
       [
      [-1  0]  [1 0]  [-w - 1   -2*w]  [ w + 1    2*w]
      [ 0 -1], [0 1], [     1  w + 1], [    -1 -w - 1]
      ],
       [
      [-1  0]  [1 0]
      [ 0 -1], [0 1]
      ],
       [
      [-1  0]  [1 0]  [ w - 1     -2]  [-w + 1      2]
      [ 0 -1], [0 1], [    -w -w + 1], [     w  w - 1]
      ],
       [
      [-1  0]  [1 0]  [-1 -1]  [ 0  1]  [ 0 -1]  [ 1  1]
      [ 0 -1], [0 1], [ 1  0], [-1 -1], [ 1  1], [-1  0]
      ],
       [
      [-1  0]  [1 0]  [-w - 1 -w + 1]  [    -w -w + 1]  [     w  w - 1]
      [ 0 -1], [0 1], [     1      w], [     1  w + 1], [    -1 -w - 1],

      [w + 1 w - 1]
      [   -1    -w]
      ]]

    .. NOTE::
  
       An edge is said to be fixed if both of its end points are fixed by the matrix. The order of the list is the one produced by the member function edges_list(), which allows to identify the corresponding vertex.

    .. SEEALSO::

      :function: `edges_list`

    """   

    return self._access('edge_stabilizer')
  
  def identification_matrices(self):
    """
    Returns the list of matrices of the Bianchi group which identifies two points in the list of vertices of the Bianchi Fundamental Polyhedron.  The imaginary part of the coordinates in the complex plane is represented using the imaginary quadratic base element `w` in the imaginary quadratic number field. This base element is saved as `B.w` if `B` is our Bianchi orbifold object.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.identification_matrices()
      [[
      [-1  0]  [1 0]  [ 0  1]  [ 0 -1]
      [ 0 -1], [0 1], [-1  0], [ 1  0]
      ],
       [
      [-1  0]  [1 0]  [-w  1]  [ w -1]
      [ 0 -1], [0 1], [ 1  w], [-1 -w]
      ],
       [
      [-1  0]  [1 0]  [-1 -w]  [-w  1]  [ 0  1]  [ 0 -1]  [ w -1]  [ 1  w]
      [ 0 -1], [0 1], [-w  1], [ 1  w], [-1  0], [ 1  0], [-1 -w], [ w -1]
      ],
       [
      [-1  0]  [1 0]  [-1 -2]  [ 1  2]
      [ 0 -1], [0 1], [ 1  1], [-1 -1]
      ],
       [
      [-1  0]  [1 0]  [-w - 1   -2*w]  [ w + 1    2*w]
      [ 0 -1], [0 1], [     1  w + 1], [    -1 -w - 1]
      ],
       [
      [-1  0]  [1 0]  [-w - 1   -2*w]  [-1 -2]  [ w - 1     -2]
      [ 0 -1], [0 1], [     1  w + 1], [ 1  1], [    -w -w + 1],

      [-w + 1      2]  [ 1  2]  [ w + 1    2*w]
      [     w  w - 1], [-1 -1], [    -1 -w - 1]
      ],
       [
      [-1  0]  [1 0]  [-1 -1]  [ 0  1]  [ 0 -1]  [ 1  1]
      [ 0 -1], [0 1], [ 1  0], [-1 -1], [ 1  1], [-1  0]
      ],
       [
      [-1  0]  [1 0]  [-w - 1 -w + 1]  [    -w -w + 1]  [     w  w - 1]
      [ 0 -1], [0 1], [     1      w], [     1  w + 1], [    -1 -w - 1],

      [w + 1 w - 1]
      [   -1    -w]
      ],
       [
      [-1  0]  [1 0]  [    -2 -w - 1]  [-w - 1 -w + 1]  [-w - 1     -w]
      [ 0 -1], [0 1], [-w + 1      1], [     1      w], [     2  w + 1],

      [-1 -w]  [    -1 -w - 1]  [-1 -1]  [ w - 1     -1]  [ w - 1     -2]
      [-w  1], [-w + 1      2], [ 1  0], [-w - 1     -w], [    -w -w + 1],

      [    -w -w + 1]  [   -w     1]  [ 0  1]  [ 0 -1]  [     w     -1]
      [     1  w + 1], [w + 1 w - 1], [-1 -1], [ 1  1], [-w - 1 -w + 1],

      [     w  w - 1]  [-w + 1      2]  [-w + 1      1]  [ 1  1]
      [    -1 -w - 1], [     w  w - 1], [ w + 1      w], [-1  0],

      [    1 w + 1]  [ 1  w]  [ w + 1      w]  [w + 1 w - 1]  [    2 w + 1]
      [w - 1    -2], [ w -1], [    -2 -w - 1], [   -1    -w], [w - 1    -1]
      ],
       [
      [-1  w]  [-w -1]  [ w  1]  [ 1 -w]
      [ 0 -1], [ 1  0], [-1  0], [ 0  1]
      ],
       [
      [-1 -w]  [ 0  1]  [ 0 -1]  [1 w]
      [ 0 -1], [-1 -w], [ 1  w], [0 1]
      ],
       [
      [-1  1]  [-1 -1]  [ 1  1]  [ 1 -1]
      [ 0 -1], [ 1  0], [-1  0], [ 0  1]
      ],
       [
      [-1 -1]  [ 0  1]  [ 0 -1]  [1 1]
      [ 0 -1], [-1 -1], [ 1  1], [0 1]
      ],
       [
      [-w - 1     -1]  [   -1 w + 1]  [     1 -w - 1]  [w + 1     1]
      [     1      0], [    0    -1], [     0      1], [   -1     0]
      ],
       [
      [    -1 -w - 1]  [     0      1]  [    0    -1]  [    1 w + 1]
      [     0     -1], [    -1 -w - 1], [    1 w + 1], [    0     1]
      ],
       [
      [    -1 -w + 1]  [    -1 -w - 1]  [    1 w + 1]  [    1 w - 1]
      [     0     -1], [     1      w], [   -1    -w], [    0     1]
      ],
       [
      [   -1 w - 1]  [    -w -w - 1]  [    w w + 1]  [     1 -w + 1]
      [    0    -1], [     1      1], [   -1    -1], [     0      1]
      ],
       [
      [-w - 1 -w + 1]  [-1  1]  [ 1 -1]  [w + 1 w - 1]
      [     1      w], [ 0 -1], [ 0  1], [   -1    -w]
      ],
       [
      [-1 -1]  [    -w -w + 1]  [     w  w - 1]  [1 1]
      [ 0 -1], [     1  w + 1], [    -1 -w - 1], [0 1]
      ],
       [
      [-w - 1 -w + 1]  [-1  1]  [-1 -1]  [ w - 1 -w - 1]  [-w + 1  w + 1]
      [     1      w], [ 0 -1], [ 1  0], [    -w      1], [     w     -1],

      [ 1  1]  [ 1 -1]  [w + 1 w - 1]
      [-1  0], [ 0  1], [   -1    -w]
      ],
       [
      [    -1 -w - 1]  [-1 -1]  [    -w -w + 1]  [ 0  1]  [ 0 -1]
      [    -w -w + 1], [ 0 -1], [     1  w + 1], [-1 -1], [ 1  1],

      [     w  w - 1]  [1 1]  [    1 w + 1]
      [    -1 -w - 1], [0 1], [    w w - 1]
      ],
       [
      [-w - 1 -w - 2]  [-1  w]  [ 1 -w]  [w + 1 w + 2]
      [     1      1], [ 0 -1], [ 0  1], [   -1    -1]
      ],
       [
      [-1 -w]  [    -1 -w - 2]  [     1  w + 2]  [1 w]
      [ 0 -1], [     1  w + 1], [    -1 -w - 1], [0 1]
      ],
       [
      [-w - 1     -1]  [-1  w]  [    -w -w - 1]  [    w w + 1]  [ 1 -w]
      [     1      0], [ 0 -1], [     1      1], [   -1    -1], [ 0  1],

      [w + 1     1]
      [   -1     0]
      ],
       [
      [-1 -w]  [    -1 -w - 1]  [     0      1]  [    0    -1]
      [ 0 -1], [     1      w], [    -1 -w - 1], [    1 w + 1],

      [    1 w + 1]  [1 w]
      [   -1    -w], [0 1]
      ]]

    .. NOTE::
  
      This identification is calculated only for those pairs of vertices which are in same orbit. The indexing is defined according to the matrices listed in the member function transport_from() and corresponding vertices identified are listed in the function equivalent_vertices. For example [1, 10, 12, 14] is the list of indices of matrices in the function identification_matrices which identifies the list of vertices [1, 2, 4, 5]. This indexing of vertices is according to the function of total_point_set. 

    .. SEEALSO::

      :function: `total_point_set`
      :function: `transport_from`
      :function: `equivalent_vertices`
  
    """   

    return self._access('identification_matrices')

  def equivalent_vertices(self):
    """ 
    Returns the list of indices of vertices which are identified by a matrix in the Bianchi group.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.equivalent_vertices()  
      [[1, 2, 4, 5],
       [2, 1, 4, 5],
       [3, 6],
       [4, 1, 2, 5],
       [5, 1, 2, 4],
       [6, 3],
       [7, 8],
       [8, 7],
       [9]]

    .. NOTE::
  
      This indexing is defined according to the points in the function total_point_set. You can use indexing in matrices to get the list of equivalent vertices for
       a specific vertex.

    .. SEEALSO::

      :function: `total_point_set`

    """     

    return self._access('equivalent_vertices')

  def transport_from(self):
    """
    Returns a list of indices of matrices. The indices in this list are according to the index in the function identification_matrices.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.transport_from()
      [[1, 10, 12, 14],
       [2, 11, 16, 18],
       [3, 20],
       [4, 13, 17, 22],
       [5, 15, 19, 23],
       [6, 21],
       [7, 24],
       [8, 25],
       [9]]

    .. NOTE::
  
      This indexing is defined according to vertices order defined in the function total_point_set. You can use indexing in matrices to get the list of matrices 
      which identifies a particular vertex to a set of vertex which can be obtained from the function equivalent_matrices.

    .. SEEALSO::

      :function: `total_point_set`
      :function: `identification_matrices`
      :function: `equivalent_vertices`


    """   

    return self._access('transport_from')

  def equivalent_edges(self):
    """ 
    Returns the list of indices of edges which are identified or whose end points are in the same orbit.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.equivalent_edges()
      [[6],
       [4, 7, 9],
       [8],
       [2, 7, 9],
       [10],
       [1],
       [2, 4, 9],
       [3],
       [2, 4, 7],
       [5],
       [],
       []]

    .. NOTE::
  
      This indexing is defined according to edges order defined in the function edges_list. You can use indexing in matrices to get the list of edges 
      which are identified with a specific edge. The empty vector represents that indexed edge which is not identified with any edges in the edges list.

    .. SEEALSO::

      :function: `edges_list`

    """   

    return self._access('equivalent_edges')

  def edges_of_2cell(self):
    """ 
    Returns the list of indices of edges which forms a 2-cell in the Bianchi fundamental polyhedron indexed according to the hemispheres which forms a 2-cell listed 
    in the function two_cell_support.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.edges_of_2cell()
      [[5, 11, 1, 2], [5, 12, 3, 4], [0], [0], [0], [0]]

    .. NOTE::
  
      This indexing is defined according to edges order defined in the function edges_list. You can use indexing in matrices to get the list of edges 
      of a particular 2 cell. The list with 0 represents that the function recorded other 2 cells but they were deleted according to Swan's criterion.

    .. SEEALSO::

      :function: `edges_list`
      :function: `two_cell_support`

    """   

    return self._access('edges_of_2cell')

  def edge_origin(self):
    """ 
    Returns a list of indices of vertices through which an edge of the convex polygon defined for Bianchi Fundamental Polyhedron originates. It helps in providing a 
    orientation for the edge by combining with the list generated through the function edge_end. 

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.edge_origin()  
      [1, 1, 2, 2, 3, 4, 4, 5, 5, 6, 7, 8]

    .. NOTE::
  
      This indexing is defined according to edges order defined in the function edges_list. You can use indexing in matrices to get the vertex through
      which a particular edge originate. Vertex indexing is according to the order defined in the function total_point_set.

    .. SEEALSO::

      :function: `edges_list`
      :function: `total_point_set`

    """
    
    return self._access('edge_origin')

  def edge_end(self):
    """ 

    Returns a list of indices of vertices through which an edge of the convex polygon defined for Bianchi Fundamental Polyhedron ends. It helps in providing a 
    orientation for the edge by combining with the list generated through the function edge_origin. 

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.edge_end() 
      [3, 7, 3, 8, 9, 6, 7, 6, 8, 9, 9, 9]

    .. NOTE::
  
      This indexing is defined according to edges order defined in the function edges_list. You can use indexing in matrices to get the vertex through
      which a particular edge ends. Vertex indexing is according to the order defined in the function total_point_set.

    .. SEEALSO::

      :function: `edges_list`
      :function: `total_point_set`

    """
    
    return self._access('edge_end')

  def vertex_orbit_number(self):
    """ 
    Returns the orbit number or label for the set of vertices.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.vertex_orbit_number()  
      [1, 1, 2, 1, 1, 2, 3, 3, 4]

    .. NOTE::
  
      This indexing is defined according to vertices order defined in the function total_point_set. You can use indexing in matrices to get the orbit in which
      a particular vertex lies. Orbits are labeled or indexed according to the the function vertex_orbit_representative. 

    .. SEEALSO::

      :function: `vertex_orbit_representative`
      :function: `total_point_set`

    """
    
    return self._access('vertex_orbit_number')

  def vertex_orbit_representative(self):
    """ 
    Returns a list of indices of vertices in which each index represents a vertex contained in that indexed orbit.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.vertex_orbit_representative()
      [1, 3, 7, 9, 0, 0, 0, 0]

    .. NOTE::
  
      This indexing is defined according to the orbit number as defined in function vertex_orbit_number. You can use indexing in matrices to get a vertex in a
      particular orbit. This list is defined longer in length than required and hence the remaining values are filled with 0's. Vertex indexing is according to 
      the order defined in the function total_point_set.

    .. SEEALSO::

      :function: `vertex_orbit_number`
      :function: `total_point_set`

    """
    
    return self._access('vertex_orbit_representative')

  def number_of_vertex_orbits(self):
    """ 
    Returns the number of partitions of the vertex set done through identifying vertices using matrices from the Bianchi group.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.number_of_vertex_orbits()
      4

    .. SEEALSO::

      :function: `equivalent_vertices`
      :function: `vertex_orbit_number`
      :function: `vertex_orbit_representative`
      :function: `total_point_set`

    """
    
    return self._access('number_of_vertex_orbits')

  def number_of_edge_orbits(self):
    """ 
    Returns the number of partitions of the edge set done through identifying vertices using matrices from the Bianchi group.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.number_of_edge_orbits()
      6

    .. SEEALSO::

      :function: `edges_list`
      :function: `equivalent_edges`

    """
    
    return self._access('number_of_edge_orbits')

  def edge_orbit_number(self):
    """ 
    Returns the orbit number or label for the set of edges.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.edge_orbit_number()  
      [1, 2, 3, 2, 4, 1, 2, 3, 2, 4, 5, 6, 0, 0, 0, 0]

    .. NOTE::
  
      This indexing is defined according to edges order defined in the function edges_list. You can use indexing in matrices to get the orbit in which
      a particular edge lies. Orbits are labeled or indexed according to the the function edge_orbit_representative. This list is defined longer in length than         required and hence the remaining values are filled with 0's

    .. SEEALSO::

      :function: `edge_orbit_representative`
      :function: `edges_list`

    """
    
    return self._access('edge_orbit_number')

  def edge_orbit_representative(self):
    """ 
    Returns a list of indices of edges in which each index represents an edge contained in that indexed orbit.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.edge_orbit_representative()
      [1, 2, 3, 5, 11, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0]

    .. NOTE::
  
      This indexing is defined according to the orbit number as defined in function edge_orbit_number. You can use indexing in matrices to get an edge in a
      particular orbit. This list is defined longer in length than required and hence the remaining values are filled with 0's. Edge indexing is according to 
      the order defined in the function edges_list.

    .. SEEALSO::

      :function: `edge_orbit_number`
      :function: `edges_list`

    """
    
    return self._access('edge_orbit_representative')

  def mu(self):
    """ 
    Returns the changing value of Mu, an element in the ring of integers of the imaginary quadratic field, till it reaches an expected value, starting with smallest
    value which the norm of a non-zero element in the ring of integers can take i.e. 1. The radius of the hemispheres required to define Bianchi Fundamental 
    Polyhedron is 1/Mu.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.mu()
      [1, 1, 1, 1]

    """
    
    return self._access('mu')

  def Lambda(self):
    """ 
    Returns the changing value of Lambda, an element in the ring of integers of the imaginary quadratic field, till it reaches an expected value. It is changed 
    according to the change in Mu so that the center of hemisphere which Lambda/Mu remains in the fundamental rectangle.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.Lambda()
      [1, 1, 1, 1]

    """
    
    return self._access('Lambda')

  def number_of_spheres(self):
    """
    Returns the number of hemispheres which are required to define the Bianchi Fundamental Polyhedron.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.number_of_spheres()
      4

    """
    
    return self._access('number_of_spheres')

  def corners_of_2cell(self):
    """ 
    Returns the vertices of the 2-cells defined in the cell structure for the boundary of the Bianchi Fundamental Polyhedron.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.corners_of_2cell()
      [[1, 3, 7, 9], [2, 8, 3, 9], 0, 0, 0, 0]

    .. NOTE::
  
      This indexing is defined according to the 2-cell listed in function two_cell_support. You can use indexing in matrices to get the corners of a
      particular 2-cell. This list is defined longer in length than required and hence the remaining values are filled with 0's. Vertex indexing is according to 
      the order defined in the function total_point_set.

    .. SEEALSO::

      :function: `two_cell_support`
      :function: `total_point_set`
    """
    
    return self._access('corners_of_2cell')

  def delete_cell_flag(self):
    """ 
    Returns whether a particular 2-cell originally defined for a hemisphere has been deleted or not when the quotient space is formed.

    EXAMPLE:::

      sage:A = BianchiOrbifold(2)
      sage:A.delete_cell_flag()
      [0, 0, 0, 0, 0, 0]

    .. NOTE::
  
      This indexing is defined according to the hemispheres listed in function sphere_center. In the starting we put a 2-cell onto each kept hemisphere. This list 
      is defined longer in length than required and hence the remaining values are filled with 0's. The significant length of the list is equal to the number of  
      hemispheres required to define the Bianchi Fundamental Polyhedron which can be obtained from the function number_of_spheres. 0 here represents not deleted.

    .. SEEALSO::

      :function: `number_of_spheres`
      :function: `sphere_center`

    """
    
    return self._access('delete_cell_flag')

  def boundary_matrix(self):
    """
    Returns a matrix which tells for a particular edge in an edge orbit, the information about the vertex orbit number of the origin and end points of the edge.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.boundary_matrix()
      [-1 -1 -1  0  0  0]
      [ 1  0  1 -1  0  0]
      [ 0  1  0  0 -1 -1]
      [ 0  0  0  1  1  1]

    .. NOTE::
  
      This is a matrix of size [number_of_vertex_orbits][number_of_edge_orbits]. If in a j'th column there is an entry -1 it represents that the edge originates in         that i'th vertex orbit number and an entry +1 represents that the edge ends in that k'th vertex orbit number.

    .. SEEALSO::

      :function: `number_of_vertex_orbits`
      :function: `number_of_edge_orbits`
      :function: `edge_orbit_representative`
      :function: `edge_orbit_number`
      :function: `edges_list`
      :function: `equivalent_edges`

    """
    
    return self._access('boundary_matrix')
  
  def number_of_2cells(self):
    """
    Returns the number of 2-cells present in the cell structure of the boundary of the Bianchi Fundamental Polyhedron.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.number_of_2cells()
      2

    .. SEEALSO::

      :function: `two_cell_support`
  
    """
    
    return self._access('number_of_2cells')

  def cell_boundary_matrix(self):
    """ 
    Returns a matrix which tells for a particular 2-cell in the boundary of the Bianchi Fundamental Polyhedron, the information about the edge orbit number and the 
    orientation of the edge.

    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.cell_boundary_matrix()
      [-1  0]
      [ 1 -1]
      [ 0  1]
      [-1  1]
      [ 1  0]
      [ 0 -1]

    .. NOTE::
  
      This is a matrix of size [number_of_edge_orbits][number_of_two_cells]. If in a j'th column there is an entry -1 it represents that the edge in the 2-cell         is present in the i'th edge orbit and its orientation is different than the entries which are +1. Edge Orientations are defined according to a determinant
      of a matrix formed from the entries of components of edge origin, barycenter, edge end.

    .. SEEALSO::

      :function: `number_of_two_cells`
      :function: `number_of_edge_orbits`

    """
    
    return self._access('cell_boundary_matrix')

  def class_number(self):
    """ 
    Returns the class number for the ring of integers defined for the imaginary quadratic field for a particular value of m.


    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.class_number()
      1

    """
    
    return self._access('class_number')

  def edge_two_dim(self):
    """ 
    Returns the number of elements which contributes to the 2-primary part of E^1_{1,1}. 


    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.edge_two_dim()
      3

    .. SEEALSO::

      :function: `edge_2label`

    """
    
    return self._access('edge_two_dim')

  def edge_2_label(self):
    """ 
    Returns the edge orbit numbers which contributes to the 2-primary part of E^1_{1,1}.


    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.edge_2_label()
      [1, 3, 4]

    """
    
    return self._access('edge_2_label')

  def edge_three_dim(self):
    """ 
    Returns the number of elements which contributes to the 3-primary part of E^1_{1,1}.  


    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.edge_three_dim()
      2

    .. SEEALSO::

      :function: `edge_3label`
    """
    
    return self._access('edge_three_dim')

  def edge_3_label(self):
    """ 
    Returns the edge orbit numbers which contributes to the 3-primary part of E^1_{1,1}.


    EXAMPLE::

      sage:A = BianchiOrbifold(2)
      sage:A.edge_3_label()
      [5, 6]  

    """
    
    return self._access('edge_3_label')

  @staticmethod
  def _gptosagerectmatrixConverter(c):

    d = matrix(c[0])
    for i in range(1,len(c)):
      d = d.insert_row(i,c[i])    
    d = d.transpose()
    
    return d      

  @staticmethod
  def _gptosagerectmatrixConverter(c):

    d = matrix(c[0])
    for i in range(1,len(c)):
      d = d.insert_row(i,c[i])    
    d = d.transpose()
    
    return d      

  @staticmethod
  def _gptosagematrixConverter(c):

    for i in range(0,len(c)):
      for j in range(0,len(c[i])):
        c[i][j] = matrix([c[i][j][0],c[i][j][1]]).transpose()
    
    return c      

  @staticmethod
  def _known_limit(m):
    # Pick the value based on computational experience
    known = {1: 1, 2: 4, 3: 1, 5: 25, 6: 34, 7: 1, 10: 66, 11: 6, 13: 86, 14: 108,
             15: 17, 17: 134, 19: 10, 21: 192, 22: 226, 23: 18, 26: 300, 29: 342,
             30: 386, 31: 30, 32: 177, 33: 1091, 34: 482, 35: 41, 36: 226, 37: 534,
             38: 588, 39: 49, 41: 646, 43: 22, 47: 49, 51: 64, 55: 86, 58: 1282,
             59: 39, 67: 34, 68: 134, 71: 93, 79: 95, 83: 49, 87: 121, 91: 100,
             95: 121, 103: 85, 107: 121, 111: 177, 115: 144, 123: 209, 139: 103,
             159: 495, 163: 82, 167: 739, 179: 161, 187: 209, 193: 9842, 211: 121,
             231: 617, 235: 324, 251: 300, 267: 801, 283: 208, 307: 207, 331: 190,
             379: 217, 403: 484, 427: 576, 499: 468, 547: 488, 571: 1200, 643: 341,
             667: 801, 715: 2081, 763: 1409, 883: 689, 907: 704, 1027: 1296,
             1051: 830, 1507: 2409, 1879: 3661}
    if m in known:
      return known[m]

  @staticmethod
  def _estimate_limit(m):
    # Use an estimation formula, based on extrapolation of the above values
    K = QuadraticField(-m, 'w')
    idealClassNumber = K.class_number()

    if m % 4 != 3:
      # discriminant = -4m
      limit = ceil(3*sqrt(2*idealClassNumber)*4*m);
    else:
      # discriminant = -m
      limit = ceil(3*sqrt(idealClassNumber/2)*m);

    return limit


