from sage.all import *
from .util import E
from collections import namedtuple
from .algebra import Algebra, tensor_with_duals

import json

from pkg_resources import resource_exists, resource_filename

__db_filename = 'p-adic-examples.json'

if resource_exists('Zeta', __db_filename):
    with open(resource_filename('Zeta', __db_filename),'r') as f:
        _padic_raw = json.load(f)
else:
    _padic_raw = {}

ZR = namedtuple('ZetaRecord', ['names', 'input', 'subalgebra_topzeta', 'ideal_topzeta'])

_raw = [
    ########################################################################
    # Names and structure constants for many of the nilpotent and soluble Lie
    # rings below are taken from de Graaf's lists in
    #   J. Algebra 309 (2007), 2, 640--653 and Experiment. Math. 14 (2005).
    # Examples in dimension/rank 7 are taken from Seeley's tables in
    #   Trans. Amer. Math. Soc. 335 (1993), 2, 479--496. 
    # Some examples are lifted straight from Woodward's archive at
    #   http://www.lack-of.org.uk/zfarchive/.
    ########################################################################

    ####################################
    # RANK 1
    ####################################


    ZR( ['L(1,1)', '(ZZ,0)', 'C(1,1)'],
        { 'rank' : 1, 'descr' : 'ZZ with zero multiplication' },
        '1/s',
        '1/s'
        ),

    ####################################
    # RANK 2
    ####################################

    ZR( ['L(2,1)', '(ZZ^2,0)', 'C(2,1)'],
        { 'rank' : 2, 'descr' : 'ZZ^2 with zero multiplication' },
        '1/((s - 1)*s)',
        '1/((s - 1)*s)',
        ),

    ####################################
    # RANK 3
    ####################################
    ZR( ['L(3,1)', '(ZZ^3,0)', 'C(3,1)', 'L^1'],
        { 'rank' : 3, 'descr' : 'ZZ^3 with zero multiplication' },
        '1/((s - 1)*(s - 2)*s)',
        '1/((s - 1)*(s - 2)*s)'
        ),

    ZR( ['L(3,2)', 'Heisenberg', 'h(ZZ)', 'H(2)', 'L^4(0)', 'n(3,ZZ)', 'A(3,5)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 0 ],
                        [-3, 0, 0 ],
                        [ 0, 0, 0 ] ],
            'descr': 'Heisenberg Lie ring over ZZ',
            },
        '3/2/((2*s - 3)*(s - 1)*s)',
        '1/((3*s - 2)*(s - 1)*s)'
        ),

    ####################################
    # RANK 4
    ####################################

    ZR( ['L(4,1)', '(ZZ^4,0)', 'C(4,1)', 'M^1'],
        { 'rank' : 4, 'descr' : 'ZZ^4 with zero multiplication' },
        '1/((s - 1)*(s - 2)*(s - 3)*s)',
        '1/((s - 1)*(s - 2)*(s - 3)*s)'
        ),

    ZR( ['L(4,2)', 'L(3,2) (+) (ZZ,0)', 'h(ZZ) (+) (ZZ,0)', 'H(2) (+) (ZZ,0)', 'M^5'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 0, 0 ],
                        [-3, 0, 0, 0 ],
                        [ 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0 ] ],
            'descr': 'Direct sum of the Heisenberg Lie ring and (ZZ,0)',
            },
        '1/2*(3*s - 4)/((2*s - 3)*(s - 1)*(s - 2)^2*s)',
        '1/3/((s - 1)^2*(s - 2)*s)',
        ),
    
    ZR( ['L(4,3)', 'M3', 'nb(sp(4,ZZ))', 'M^7(0,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0 ],
                        [-3, 0, 0, 0 ],
                        [-4, 0, 0, 0 ],
                        [ 0, 0, 0, 0 ] ],
            },
        '1/2*(13*s - 15)/((3*s - 5)*(2*s - 3)^2*(s - 1)*s)',
        '1/2*(5*s - 2)/((5*s - 3)*(3*s - 2)*(2*s - 1)*(s - 1)*s)'
        ),


    ####################################
    # RANK 5
    ####################################
    
    ZR( ['L(5,1)', '(ZZ^5,0)', 'C(5,1)'],
        { 'rank' : 5, 'descr' : 'ZZ^5 with zero multiplication' },
        '1/((s - 1)*(s - 2)*(s - 3)*(s - 4)*s)',
        '1/((s - 1)*(s - 2)*(s - 3)*(s - 4)*s)'
        ),

    ZR( ['L(5,2)', 'h(ZZ) (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            'descr': 'Direct sum of the Heisenberg Lie ring and (ZZ^2,0)',
            },
        '1/2*(3*s - 5)/((2*s - 5)*(s - 1)*(s - 2)^2*(s - 3)*s)',
        '1/((3*s - 4)*(s - 1)*(s - 2)*(s - 3)*s)'
        ),

    ZR( ['L(5,3)', 'M3 (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 0 ],
                        [-3, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/24*(13*s^2 - 41*s + 32)/((s - 1)*(s - 2)^5*s)',
        '1/3*(5*s - 3)/((5*s - 4)*(4*s - 3)*(s - 1)^2*(s - 2)*s)'
        ),

    ZR( ['L(5,4)', 'h(ZZ) curlyvee h(ZZ)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 5, 0 ],
                        [ 0, 0,-5, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            'descr': 'Central product of two copies of the Heisenberg Lie ring',
            },
        '1/3*(17*s - 21)/((3*s - 4)*(3*s - 7)*(s - 1)*(s - 2)*(s - 3)*s)',
        '1/((5*s - 4)*(s - 1)*(s - 2)*(s - 3)*s)'
        ),

    ZR( ['L(5,5)', 'Q5'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 0 ],
                        [-3, 0, 0, 5, 0 ],
                        [-5, 0, 0, 0, 0 ],
                        [ 0,-5, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/18*(153*s^3 - 704*s^2 + 1064*s - 528)/((6*s - 11)*(3*s - 4)*(s - 1)*(s - 2)^4*s)',
        '1/3/((5*s - 4)*(s - 1)^2*(s - 2)*s)'
        ),

    ZR( ['L(5,6)', 'Fil4'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0 ],
                        [-3, 0, 5, 0, 0 ],
                        [-4,-5, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/3*(392031360*s^9 - 5741480808*s^8 + 37286908278*s^7 - 140917681751*s^6 +'
        '341501393670*s^5 - 550262853249*s^4 + 589429290044*s^3 -'
        '404678115300*s^2 + 161557332768*s - 28569052512)/('
        '(15*s - 26)*(7*s - 12)*(7*s - 13)*(6*s - 11)^3*(5*s - 8)*(5*s - 9)*'
        '(4*s - 7)^2*(3*s - 4)*(2*s - 3)*(s - 1)*s)',
        '1/2*(5040*s^4 - 10157*s^3 + 7522*s^2 - 2412*s + 280)/('
        '(12*s - 7)*(8*s - 5)*(7*s - 4)*(5*s - 2)*(5*s - 3)*'
        '(3*s - 2)^2*(s - 1)*s)'
        ),

    ZR( ['L(5,7)', 'M4'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0 ],
                        [-3, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/2*(46*s^3 - 216*s^2 + 333*s - 168)/((7*s - 12)*(4*s - 7)*(2*s - 3)*(s - 1)*(s - 2)^3*s)',
        '2*(1386*s^3 - 2023*s^2 + 948*s - 140)*(2*s - 1)/((12*s - 7)*(11*s - 6)*(8*s - 5)*(7*s - 4)*(5*s - 2)*(3*s - 2)^2*(s - 1)*s)',
        ),

    ZR( ['L(5,8)', 'G3'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0 ],
                        [-4, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/6*(7*s - 10)/((2*s - 5)*(s - 1)*(s - 2)^3*s)',
        '2/((5*s - 6)*(3*s - 4)*(s - 1)*(s - 2)*s)'
        ),

    ZR( ['L(5,9)', 'F(3,2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 0 ],
                        [-3, 0, 5, 0, 0 ],
                        [-4,-5, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            'descr': 'Free nilpotent Lie ring of class 3 on 2 generators',
            },
        '1/6*(121*s^3 - 565*s^2 + 864*s - 432)/((5*s - 8)*(5*s - 9)*(2*s - 3)*(s - 1)*(s - 2)^3*s)',
        '2*(35*s^2 - 42*s + 12)/((7*s - 6)*(5*s - 4)^2*(4*s - 3)*(3*s - 2)*(s - 1)*s)'
        ),

    ####################################
    # RANK 6
    ####################################

    ZR( ['L(6,1)', '(ZZ^6,0)', 'C(6,1)'],
        { 'rank' : 6, 'descr' : 'ZZ^6 with zero multiplication' },
        '1/((s - 1)*(s - 2)*(s - 3)*(s - 4)*(s - 5)*s)',
        '1/((s - 1)*(s - 2)*(s - 3)*(s - 4)*(s - 5)*s)',
        ),

    ZR( ['L(6,2)', 'h(ZZ) (+) (ZZ^3,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 0, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            'descr': 'Direct sum of the Heisenberg Lie ring and (ZZ^3,0)',
            },
        '3/2/((2*s - 5)*(s - 1)*(s - 3)^2*(s - 4)*s)',
        '1/((3*s - 5)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)'
        ),

    ZR( ['L(6,3)', 'M3 (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/2*(13*s - 28)*(3*s - 5)/((3*s - 7)*(3*s - 8)*(2*s - 5)^2*(s - 1)*(s - 2)*(s - 3)*s)',
        '1/20*(5*s - 4)/((3*s - 4)*(s - 1)^3*(s - 2)*(s - 3)*s)'
        ),

    ZR( ['L(6,4)', 'h(ZZ) curlyvee h(ZZ) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 5, 0, 0 ],
                        [ 0, 0,-5, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            'descr': 'Direct sum of (ZZ,0) and the central product of two Heisenberg Lie rings', 
            },
        '(17*s^2 - 63*s + 56)/((3*s - 5)*(3*s - 7)*(3*s - 8)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)',
        '1/5/((s - 1)^2*(s - 2)*(s - 3)*(s - 4)*s)'
        ),

    ZR( ['L(6,5)', 'Q5 (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 0, 0 ],
                        [-3, 0, 0, 5, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [ 0,-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/2*(12240*s^5 - 126784*s^4 + 521522*s^3 - 1063913*s^2 + 1075150*s - 429975)/((6*s - 13)*(5*s - 12)*(4*s - 9)*(3*s - 5)*(3*s - 7)^2*(2*s - 5)^2*(s - 1)*(s - 3)*s)',
        '1/5/((3*s - 4)*(s - 1)^2*(s - 2)*(s - 3)*s)'
        ),

    ZR( ['L(6,6)', 'Fil4 (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0 ],
                        [-3, 0, 5, 0, 0, 0 ],
                        [-4,-5, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/48*(52839554826240*s^15 - 1612571385729024*s^14 + 22945067840268288*s^13 - 201917310138409536*s^12 + 1228942670032455984*s^11 - 5479610770178424720*s^10 +'
        '18489925054934205732*s^9 - 48077179247205683304*s^8 + 97118269735864324559*s^7 - 152405042677332499112*s^6 + 184268407184801648476*s^5 - 168562287295854189878*s^4 +'
        '112921211241642321545*s^3 - 52295417007047312650*s^2 + 14969814525806597400*s - 1996549752637440000)/('
        '(15*s - 31)*(15*s - 34)*(13*s - 28)*(12*s - 25)*(9*s - 20)*(7*s - 15)*(7*s - 16)*(6*s - 13)^3*(5*s - 11)*(4*s - 9)^3*(3*s - 5)*(s - 1)*(s - 2)^4*s)', 
        '1/6*(5040*s^4 - 13845*s^3 + 14044*s^2 - 6215*s + 1008)/((9*s - 8)*(8*s - 7)*(7*s - 5)*(5*s - 3)*(5*s - 4)*(4*s - 3)*(s - 1)^2*(s - 2)*s)'
        ),

    ZR( ['L(6,7)', 'M4 (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/48*(8280*s^5 - 81688*s^4 + 320237*s^3 - 623254*s^2 + 601785*s - 230400)/((7*s - 15)*(5*s - 12)*(3*s - 8)*(2*s - 5)^2*(s - 1)*(s - 2)^4*s)',
        '4*(154*s^3 - 309*s^2 + 201*s - 42)*(3*s - 2)/((11*s - 8)*(9*s - 8)*(8*s - 7)*(7*s - 5)*(5*s - 3)*(4*s - 3)*(s - 1)^2*(s - 2)*s)',
        ),

    ZR( ['L(6,8)', 'G3 (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/2*(7*s - 12)/((3*s - 8)*(2*s - 5)*(s - 1)*(s - 2)*(s - 3)^2*s)',
        '2/((5*s - 8)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*s)'
        ),

    ZR( ['L(6,9)', 'F(3,2) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 0, 0 ],
                        [-3, 0, 5, 0, 0, 0 ],
                        [-4,-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/6*(726*s^4 - 5575*s^3 + 15887*s^2 - 19900*s + 9240)/((5*s - 11)*(5*s - 12)*(3*s - 7)*(2*s - 5)^2*(s - 1)*(s - 2)^3*s)',
        '1/30*(35*s - 24)/((7*s - 8)*(5*s - 6)*(s - 1)^3*(s - 2)*s)'
        ),

    ZR( ['L(6,10)', 'g(6,12)', 'M3 curlyvee h(ZZ)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 6, 0 ],
                        [ 0, 0, 0,-6, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/72*(166530*s^6 - 1931987*s^5 + 9294564*s^4 - 23731443*s^3 + 33913892*s^2 - 25719216*s +'
        '8087040)/((7*s - 16)*(6*s - 13)*(5*s - 12)*(4*s - 9)*(3*s - 5)*(2*s - 5)*'
        '(s - 1)*(s - 2)^3*(s - 3)*s)',
        '1/2*(7*s - 4)/((7*s - 5)*(3*s - 2)*(3*s - 4)*(s - 1)*(s - 2)*(s - 3)*s)'
        ),

    ZR( ['L(6,11)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0 ],
                        [-3, 0, 6, 0, 6, 0 ],
                        [-4,-6, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None, # subalgebras: the main loop succeeds but with >30,000 cases.
        '1/6*(63*s^2 - 79*s + 24)/((9*s - 8)*(7*s - 5)*(4*s - 3)*(3*s - 2)*(s - 1)^2*(s - 2)*s)',
        ),


    ZR( ['L(6,12)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0 ], 
                        [-3, 0, 0, 0, 6, 0 ],
                        [-4, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        # This needs to be confirmed but using an older version of Zeta, after 164 days, it proclaimed that
        # '1/144*(880210146816000*s^15 - 27086469470323200*s^14 + 388582475029069440*s^13 - 3447314863281222304*s^12 + 21149728757066494408*s^11 -
        #        95047401545553919828*s^10 + 323217032159089886774*s^9 - 846871952779323707741*s^8 + 1723657123002330212006*s^7 -
        #        2725039445294025518536*s^6 + 3318957271105819574692*s^5 - 3058049683840982443183*s^4 + 2063251282380433097544*s^3 -
        #        962268006754185428232*s^2 + 277378984012191536160*s - 37251010961708544000)/((14*s - 31)*(13*s - 28)*(11*s - 25)*(10*s - 21)^2*
        #                                                                                     (9*s - 20)*(8*s - 17)*(7*s - 16)*(6*s - 13)^2*(5*s - 11)^2*(4*s - 9)^3*(3*s - 5)*(s - 1)*(s - 2)^3*s)',
        # is the topological subalgebra zeta function.
        '1/6*(63*s^2 - 79*s + 24)/((9*s - 8)*(7*s - 5)*(4*s - 3)*(3*s - 2)*(s - 1)^2*(s - 2)*s)',
        ),

    ZR( ['L(6,13)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 6, 0 ],
                        [-3, 0, 0, 5, 0, 0 ],
                        [-5, 0, 0, 6, 0, 0 ],
                        [ 0,-5,-6, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        '1/9*(7*s - 3)/((7*s - 5)*(5*s - 4)*(2*s - 1)*(s - 1)^2*(s - 2)*s)',
        ),


    ZR( ['L(6,14)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0 ],
                        [-3, 0, 5, 0, 6, 0 ],
                        [-4,-5, 0,-6, 0, 0 ],
                        [-5, 0, 6, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['L(6,15)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 0 ],
                        [-3, 0, 5, 6, 0, 0 ],
                        [-4,-5, 0, 0, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['L(6,16)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0 ],
                        [-3, 0, 0, 0, 6, 0 ],
                        [-4, 0, 0,-6, 0, 0 ],
                        [-5, 0, 6, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['L(6,17)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 0 ],
                        [-3, 0, 6, 0, 0, 0 ],
                        [-4,-6, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['L(6,18)', 'M5'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 0 ],
                        [-3, 0, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['L(6,19;0)', 'g(6,7)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0 ],
                        [-4, 0, 0, 6, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/48*(45927*s^5 - 413991*s^4 + 1480593*s^3 - 2628249*s^2 + 2318984*s - 815360)/((7*s - 16)*(6*s - 13)*(3*s - 5)*(3*s - 7)^2*(2*s - 5)*(s - 1)*(s - 2)^3*s)',
        '1/210*(455*s^3 - 1144*s^2 + 942*s - 252)/((5*s - 6)*(4*s - 3)*(3*s - 4)*(s - 1)^4*(s - 2)*s)'
        ),

    ZR( ['L(6,19;1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0 ],
                        [-4, 0, 0, 6, 0, 0 ],
                        [-5, 0, 0, 0, 6, 0 ],
                        [ 0,-6, 0, 0, 0, 0 ],
                        [ 0, 0,-6, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        '2/3*(7*s - 3)/((7*s - 5)*(5*s - 6)*(3*s - 4)*(2*s - 1)*(s - 1)*(s - 2)*s)'
        ),

    ZR( ['L(6,19;-1)', 'n(4,ZZ)'],
        {
            'matrix_basis': [E(4,0,1), E(4,1,2), E(4,2,3),
                             E(4,0,2), E(4,1,3),
                             E(4,0,3)],
            'product':  'Lie',
            },
        None,
        '2/3*(7*s - 3)/((7*s - 5)*(5*s - 6)*(3*s - 4)*(2*s - 1)*(s - 1)*(s - 2)*s)',
        ),
   
    ZR( ['L(6,20)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 6, 0 ],
                        [-4, 0, 0, 6, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/144*(124640208000*s^10 - 2450257567200*s^9 + 21632784382760*s^8 - '
        '112945139244690*s^7 + 386146453844181*s^6 - 903233150980483*s^5 + '
        '1463738811463999*s^4 - 1622577592730987*s^3 + 1177352866364580*s^2 - '
        '504902635502400*s + 97166672064000)/((20*s - 43)*(18*s - 37)*(11*s - 24)*'
        '(10*s - 21)*(9*s - 20)*(8*s - 17)*(7*s - 15)*(5*s - 11)*(4*s - 9)*(3*s - 5)*'
        '(2*s - 5)*(s - 1)*(s - 2)^3*s)',
        '1/2*(126*s^2 - 183*s + 64)/((9*s - 8)*(7*s - 5)*(5*s - 6)*(3*s - 2)*(3*s - 4)*(s - 1)*(s - 2)*s)', 
        ),

    ZR( ['L(6,21;0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0 ],
                        [-3, 0, 5, 0, 0, 0 ],
                        [-4,-5, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['L(6,21;1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0 ],
                        [-3, 0, 5, 0, 6, 0 ],
                        [-4,-5, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['L(6,22;0)', 'L(3,2)[eps]', 'h(ZZ[X]/X^2)', 'g(6,4)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 6, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 5, 0, 0 ],
                        [ 0, 0,-5, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '(2144*s^5 - 22726*s^4 + 95508*s^3 - 198735*s^2 + 204519*s - 83160)/((5*s - 12)*(4*s - 9)*(4*s - 11)*(3*s - 5)*(3*s - 7)*(3*s - 8)*(2*s - 5)*(s - 1)*(s - 2)*(s - 3)*s)',
        '1/15*(8*s - 9)/((3*s - 4)*(2*s - 3)*(s - 1)^2*(s - 2)*(s - 3)*s)'        
        ),

    ZR( ['L(6,22;1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 6, 0, 0, 0 ],
                        [-5, 0, 0, 6, 0, 0 ],
                        [-6, 0, 0, 5, 0, 0 ],
                        [ 0,-6,-5, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['L(6,22;1) [alternative basis]', 'h(ZZ)^2', 'h(ZZ) (+) h(ZZ)', 'H(2)^2', 'H(2) (+) H(2)'],
        # NOTE: To find an isomorphism onto 'L(6,22;1)', use http://www.gap-system.org/Packages/liealgdb.html
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 6, 0, 0 ],
                        [ 0, 0,-6, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/4*(243*s^4 - 1819*s^3 + 5041*s^2 - 6145*s + 2800)/((3*s - 5)*(3*s - 7)*(3*s - 8)*(2*s - 5)^2*(s - 1)*(s - 2)^2*(s - 3)*s)',
        '1/5*(5*s - 4)/((3*s - 4)^2*(s - 1)^2*(s - 2)*(s - 3)*s)',
        ),
    
    ZR( ['L(6,23)'], # 'g(6,6)'?
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 0, 0 ],
                        [-3, 0, 0, 5, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [-6,-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/12*(1073952*s^8 - 18184536*s^7 + 134166390*s^6 - 563193911*s^5 + 1470653304*s^4 - 2445276177*s^3 +'
        '2527052978*s^2 - 1483294080*s + 378378000)/((9*s - 22)*(6*s - 13)*(5*s - 12)*(4*s - 9)*'
        '(3*s - 5)*(3*s - 7)^2*(2*s - 5)^3*(s - 1)*(s - 2)^2*s)',
        '1/30*(63*s^2 - 130*s + 66)*(15*s - 16)/((9*s - 11)*(7*s - 8)*(5*s - 6)*(3*s - 4)*(s - 1)^3*(s - 2)*s)'
        ),

    ZR( ['L(6,24;0)', 'g(6,9)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 0, 0 ],
                        [-3, 0, 6, 5, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0 ],
                        [ 0,-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/6*(7060608*s^9 - 127146024*s^8 + 1015050156*s^7 - 4714488399*s^6 + 14037015009*s^5 - 27779768957*s^4 + 36535455587*s^3 - 30785573604*s^2 + 15077423544*s -'
        '3269185920)/((9*s - 20)*(7*s - 16)*(6*s - 13)*(5*s - 11)*(4*s - 9)^3*(3*s - 5)*(3*s - 7)^2*(s - 1)*(s - 2)^3*s)',
        '1/60*(168*s^2 - 268*s + 105)/((14*s - 15)*(8*s - 7)*(s - 1)^4*(s - 2)*s)'
        ),

    ZR( ['L(6,24;1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 0, 0 ],
                        [-3, 0, 6, 5, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0 ],
                        [-6,-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['L(6,25)', 'L_W'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],

            },
        '1/2*(318*s^4 - 2654*s^3 + 8219*s^2 - 11168*s + 5600)/((5*s - 12)*(3*s - 7)*(3*s - 8)*(2*s - 5)^3*(s - 1)*(s - 2)^2*s)',
        '1/120*(585*s^2 - 1130*s + 528)/((9*s - 11)*(5*s - 6)*(3*s - 4)*(s - 1)^3*(s - 2)*s)'
        ),

    ZR( ['L(6,26)', 'F(2,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0 ],
                        [-4, 0, 6, 0, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            'descr': 'Free nilpotent Lie ring of class 2 on 3 generators',
            },
        '1/3*(25*s^2 - 94*s + 84)/((3*s - 7)*(3*s - 8)*(2*s - 5)*(s - 1)*(s - 2)^2*(s - 3)*s)',
        '2/((5*s - 8)*(3*s - 5)*(2*s - 3)*(s - 1)*(s - 2)*s)'
        ),

    ####################################
    # RANK 7
    ####################################

    ZR( ['L(7,1)', '(ZZ^7,0)', 'C(7,1)', 'L(6,1) (+) (ZZ,0)'],
        { 'rank' : 7, 'descr' : 'ZZ^7 with zero multiplication' },
        '1/((s - 1)*(s - 2)*(s - 3)*(s - 4)*(s - 5)*(s - 6)*s)',
        '1/((s - 1)*(s - 2)*(s - 3)*(s - 4)*(s - 5)*(s - 6)*s)',
        ),

    ZR( ['L(7,2)', 'L(6,2) (+) (ZZ,0)', 'h(ZZ) (+) (ZZ^4,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 0, 0, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': 'Direct sum of the Heisenberg Lie ring and (ZZ^4,0)',
            },
        '1/2*(3*s - 7)/((2*s - 7)*(s - 1)*(s - 2)*(s - 3)^2*(s - 4)*(s - 5)*s)',
        '1/3/((s - 1)*(s - 2)^2*(s - 3)*(s - 4)*(s - 5)*s)',
        ),

    ZR( ['L(7,3)', 'L(6,3) (+) (ZZ,0)', 'M3 (+) (ZZ^3,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/8*(39*s^2 - 175*s + 192)/((3*s - 8)*(3*s - 10)*(s - 1)*(s - 2)*(s - 3)^3*(s - 4)*s)',
        '5/((5*s - 6)*(4*s - 5)*(3*s - 5)*(s - 2)*(s - 3)*(s - 4)*s)',
        ),

    ZR( ['L(7,4)', 'L(6,4) (+) (ZZ,0)', 'h(ZZ) curlyvee h(ZZ) (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 5, 0, 0, 0 ],
                        [ 0, 0,-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': 'Direct sum of (ZZ^2,0) and the central product of two Heisenberg Lie rings', 
            },
        '1/9*(17*s^2 - 71*s + 72)/((3*s - 8)*(s - 1)*(s - 2)^2*(s - 3)^2*(s - 4)*(s - 5)*s)',
        '1/((5*s - 6)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*(s - 5)*s)',
        ),

    ZR( ['L(7,5)', 'L(6,5) (+) (ZZ,0)', 'Q5 (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 0, 0, 0 ],
                        [-3, 0, 0, 5, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/180*(1530*s^5 - 18842*s^4 + 92105*s^3 - 223131*s^2 + 267498*s - 126720)*(4*s - 9)/((4*s - 11)*(3*s - 8)^2*(2*s - 5)*(s - 1)*(s - 2)^2*(s - 3)^4*(s - 4)*s)',
        '1/((5*s - 6)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)',
        ),

    ZR( ['L(7,6)', 'L(6,6) (+) (ZZ,0)', 'Fil4 (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0, 0 ],
                        [-3, 0, 5, 0, 0, 0, 0 ],
                        [-4,-5, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None, # 12851 cases, life's too short!
        '1/10*(15120*s^4 - 52599*s^3 + 67714*s^2 - 38156*s + 7920)/((12*s - 11)*(9*s - 10)*(8*s - 9)*(7*s - 6)*(5*s - 4)*(3*s - 4)*(s - 1)^2*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['L(7,7)', 'L(6,7) (+) (ZZ,0)', 'M4 (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/20*(4140*s^5 - 47896*s^4 + 220249*s^3 - 502784*s^2 + 569121*s - 255150)/((7*s - 18)*(4*s - 9)*(3*s - 7)*(3*s - 10)*(2*s - 5)*(s - 1)*(s - 2)*(s - 3)^4*s)',
        '2*(1386*s^3 - 3539*s^2 + 2942*s - 792)*(6*s - 5)/((12*s - 11)*(11*s - 10)*(9*s - 10)*(8*s - 9)*(7*s - 6)*(5*s - 4)*(3*s - 4)*(s - 1)*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['L(7,8)', 'L(6,8) (+) (ZZ,0)', 'G3 (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        '7/2/((3*s - 10)*(2*s - 7)*(s - 1)*(s - 3)^2*(s - 4)*s)',
        '2/15/((s - 1)*(s - 2)^3*(s - 3)*(s - 4)*s)',
        ),

    ZR( ['L(7,9)', 'L(6,9) (+) (ZZ,0)', 'F(3,2) (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 0, 0, 0 ],
                        [-3, 0, 5, 0, 0, 0, 0 ],
                        [-4,-5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/30*(363*s^4 - 3517*s^3 + 12590*s^2 - 19696*s + 11340)/((5*s - 14)*(3*s - 8)*(2*s - 5)*(s - 1)*(s - 2)*(s - 3)^5*s)',
        '2*(35*s^2 - 76*s + 40)/((7*s - 10)*(5*s - 6)*(5*s - 8)*(4*s - 5)*(3*s - 4)*(s - 1)*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['L(7,10)', 'L(6,10) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 0, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 6, 0, 0 ],
                        [ 0, 0, 0,-6, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/12*(111020*s^8 - 2017301*s^7 + 15998618*s^6 - 72326536*s^5 + 203850500*s^4 - '
        '366777333*s^3 + 411384618*s^2 - 262971954*s + 73347120)/((7*s - 18)*(5*s - 14)*'
        '(4*s - 9)*(4*s - 11)*(3*s - 7)^2*(2*s - 5)^2*(s - 1)*(s - 2)^2*(s - 3)^2*(s - 4)*s)',
        '(7*s - 5)/((7*s - 6)*(6*s - 5)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)',
        ),

    ZR( ['L(7,22;0)', 'L(6,22;0) (+) (ZZ,0)' ],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 6, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 5, 0, 0, 0 ],
                        [ 0, 0,-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/45*(1072*s^4 - 10127*s^3 + 35461*s^2 - 54459*s + 30888)/((4*s - 11)*(4*s - 13)*(3*s - 8)*(s - 1)*(s - 2)^2*(s - 3)^3*(s - 4)*s)',
        '(8*s - 11)/((6*s - 11)*(5*s - 6)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)',
        ),

    ZR( ['L(7,23)', 'L(6,23) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 0, 0, 0 ],
                        [-3, 0, 0, 5, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-6,-5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/4320*(715968*s^8 - 14509008*s^7 + 128070788*s^6 - 642918600*s^5 + 2006747659*s^4 - '
        '3986143783*s^3 + 4918094196*s^2 - 3443712260*s + 1046960640)/((6*s - 17)*'
        '(4*s - 11)*(3*s - 8)^2*(2*s - 5)*(s - 1)*(s - 2)^2*(s - 3)^6*s)',
        '5/2*(63*s^2 - 169*s + 112)/((9*s - 14)*(7*s - 10)*(5*s - 6)*(5*s - 8)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['L(7,24;0)', 'L(6,24;0) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 0, 0, 0 ],
                        [-3, 0, 6, 5, 0, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0, 0 ],
                        [ 0,-5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/9*(35303040*s^9 - 780720552*s^8 + 7650294828*s^7 - 43587972890*s^6 + 159094465631*s^5 - '
        '385674173909*s^4 + 620769083108*s^3 - 639494815616*s^2 + 382443949880*s - 101115537600)/((9*s - 25)*'
        '(7*s - 19)*(6*s - 17)*(5*s - 14)^2*(4*s - 11)^3*(3*s - 8)^2*(2*s - 5)*(s - 1)*(s - 2)^2*(s - 3)*s)',
        '2*(252*s^3 - 845*s^2 + 936*s - 342)/((14*s - 19)*(8*s - 9)*(5*s - 6)*(4*s - 5)*(3*s - 4)^2*(s - 1)*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['L(7,25)', 'L(6,25) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],

            },
        '1/80*(636*s^4 - 6258*s^3 + 22847*s^2 - 36585*s + 21600)/((3*s - 8)*(3*s - 10)*(2*s - 5)*(s - 1)*(s - 2)*(s - 3)^5*s)',
        '1/2*(585*s^2 - 1622*s + 1120)/((9*s - 14)*(5*s - 6)*(5*s - 8)*(4*s - 5)*(3*s - 4)*(3*s - 5)*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['L(7,26)', 'L(6,26) (+) (ZZ,0)', 'F(2,3) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [[ 0, 4, 5, 0, 0, 0, 0],
                       [-4, 0, 6, 0, 0, 0, 0],
                       [-5,-6, 0, 0, 0, 0, 0],
                       [ 0, 0, 0, 0, 0, 0, 0],
                       [ 0, 0, 0, 0, 0, 0, 0],
                       [ 0, 0, 0, 0, 0, 0, 0],
                       [ 0, 0, 0, 0, 0, 0, 0]],
        },
        '1/3*(25*s^2 - 116*s + 126)/((3*s - 8)*(3*s - 10)*(2*s - 7)*(s - 1)*(s - 2)*(s - 3)^3*s)',
        '1/15/((s - 1)*(s - 2)^4*(s - 3)*s)',
        ),
    

    ######################################################################
    # 3,7
    ######################################################################

    ZR( ['g(37A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 0, 0, 0 ],
                        [-5, 0, 6, 7, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        '2/5/((7*s - 12)*(s - 1)*(s - 2)^3*(s - 3)*s)',
        ),

    ZR( ['g(37B)', 'T4'],
        {
            'simple_basis': True,
            'matrix': [[ 0, 5, 0, 0, 0, 0, 0 ],
                       [-5, 0, 6, 0, 0, 0, 0 ],
                       [ 0,-6, 0, 7, 0, 0, 0 ],
                       [ 0, 0,-7, 0, 0, 0, 0 ],
                       [ 0, 0, 0, 0, 0, 0, 0 ],
                       [ 0, 0, 0, 0, 0, 0, 0 ],
                       [ 0, 0, 0, 0, 0, 0, 0 ]],
            },
        '1/144*(11700*s^7 - 193524*s^6 + 1365106*s^5 - 5323213*s^4 + 12394047*s^3 - 17233680*s^2 + 13256364*s - 4354560)/((5*s - 14)*(3*s - 8)^2*(2*s - 5)^2*(s - 1)*(s - 2)^2*(s - 3)^5*s)',
        '(255*s^3 - 1031*s^2 + 1373*s - 600)/((7*s - 12)*(5*s - 6)*(5*s - 8)*(3*s - 5)^3*(s - 1)*(s - 2)*(s - 3)*s)'
        ),

    ZR( ['g(37C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 0, 0, 0 ],
                        [-5, 0, 6, 7, 0, 0, 0 ],
                        [ 0,-6, 0, 5, 0, 0, 0 ],
                        [ 0,-7,-5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },

        '1/180*(41532*s^6 - 609721*s^5 + 3703235*s^4 - 11901991*s^3 + 21330141*s^2 - 20190156*s + 7876440)/((6*s - 17)*(4*s - 11)*(4*s - 13)*(3*s - 8)*(2*s - 5)*(s - 1)*(s - 2)^2*(s - 3)^4*s)',
        None # exceeded depth
        ),

    ZR( ['g(37D)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 7, 0, 0, 0, 0 ],
                        [-5, 0, 0, 7, 0, 0, 0 ],
                        [-7, 0, 0, 6, 0, 0, 0 ],
                        [ 0,-7,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,# singular
        None # exceeded depth
        ),


    ######################################################################
    # 3,5,7
    ######################################################################

    ZR( ['g(357A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 7, 0, 0, 0 ],
                        [-3, 0, 0, 6, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None # toric singularities
        ),

    ZR( ['g(357B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 7, 0, 0, 0 ],
                        [-3, 0, 6, 0, 0, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None # toric singularities
        ),

    ZR( ['g(357C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 7, 0, 0, 0 ],
                        [-3, 0, 6, 5, 0, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0, 0 ],
                        [-7,-5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ######################################################################
    # 2,7
    ######################################################################

    ZR( ['g(27A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 0, 7, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 7, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/36*(839*s^4 - 7830*s^3 + 27217*s^2 - 41718*s + 23760)/((4*s - 11)*(3*s - 8)*(2*s - 5)^2*(s - 1)*(s - 2)*(s - 3)^3*(s - 4)*s)',
        '2*(4*s - 5)/((7*s - 10)*(5*s - 6)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)',
        ),

    ZR( ['g(27B)'],
        {
            'simple_basis': True,
            'matrix':[ [ 0, 6, 0, 0, 7, 0, 0 ],
                       [-6, 0, 7, 0, 0, 0, 0 ],
                       [ 0,-7, 0, 6, 0, 0, 0 ],
                       [ 0, 0,-6, 0, 0, 0, 0 ],
                       [-7, 0, 0, 0, 0, 0, 0 ],
                       [ 0, 0, 0, 0, 0, 0, 0 ],
                       [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },

        '1/9*(793*s^4 - 7182*s^3 + 24093*s^2 - 35398*s + 19152)/((7*s - 19)*(5*s - 14)*(3*s - 8)*(2*s - 5)*(s - 1)*(s - 2)^2*(s - 3)^2*(s - 4)*s)', # 11,556 toric data, 2,617,287,896 SURFs, 11.2 GB, 29 days
        '2/((7*s - 10)*(5*s - 6)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)',
        ),


    ######################################################################
    # 2,5,7
    ######################################################################

     ZR( ['g(257A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 0, 7, 0, 0 ],
                        [-3, 0, 0, 6, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
         '1/36*(16879275*s^8 - 335882855*s^7 + 2915881445*s^6 - 14419964396*s^5 + 44417109503*s^4 - 87230055182*s^3 + 106615575230*s^2 - 74108459340*s + 22415500800)/((7*s - 19)^2*(7*s - 20)*(5*s - 14)^2*(4*s - 11)*(3*s - 8)^2*(2*s - 5)^2*(s - 1)*(s - 2)*(s - 3)^2*s)',
        '3/((7*s - 9)*(5*s - 8)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*s)',
        ),

     ZR( ['g(257B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 0, 7, 0, 0 ],
                        [-3, 0, 0, 7, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
         # NOTE: Woodward uses slightly different structure constants
         '1/288*(9066960000*s^11 - 248494313760*s^10 + 3090910452008*s^9 - 23030782078062*s^8 + 114208844521753*s^7 - 395733704421642*s^6 + 977554548042132*s^5 - 1721295328640052*s^4 + 2116927954575871*s^3 - 1731521948910612*s^2 + 847571544397964*s - 188051108044800)/((7*s - 19)^2*(5*s - 13)*(5*s - 14)^2*(4*s - 11)^3*(3*s - 8)^3*(2*s - 5)^3*(s - 1)*(s - 2)*(s - 3)*s)', # no. of surfs = 3,384,788,094 (14.2 GB)
         '1/36*(2976*s^4 - 13674*s^3 + 23344*s^2 - 17515*s + 4860)/((7*s - 9)*(5*s - 6)*(4*s - 5)^2*(3*s - 4)*(2*s - 3)*(s - 1)^2*(s - 2)*(s - 3)*s)',
        ),

     ZR( ['g(257C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 0, 0, 0, 0 ],
                        [-3, 0, 0, 6, 7, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
         '1/72*(6562080*s^8 - 127086360*s^7 + 1073900260*s^6 - 5170500064*s^5 + 15510346136*s^4 - 29676516593*s^3 + 35356447047*s^2 - 23972124906*s + 7078579200)/((8*s - 21)*(7*s - 19)*(5*s - 14)*(4*s - 11)*(3*s - 8)^3*(2*s - 5)^3*(s - 1)*(s - 2)*(s - 3)^2*s)',
        '3/((7*s - 9)*(5*s - 8)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['g(257D)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 7, 0, 0, 0 ],
                        [-3, 0, 0, 6, 7, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(257E)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 0, 0, 0, 0 ],
                        [-3, 0, 0, 7, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 6, 0, 0 ],
                        [ 0, 0, 0,-6, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(257F)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 7, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 6, 0, 0 ],
                        [ 0, 0, 0,-6, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(257G)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 0, 7, 0, 0 ],
                        [-3, 0, 0, 7, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 6, 0, 0 ],
                        [-7, 0, 0,-6, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(257H)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 0, 0, 0, 0 ],
                        [-3, 0, 0, 6, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 7, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(257I)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 6, 7, 0, 0 ],
                        [-3, 0, 7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(257J)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 0, 7, 0, 0 ],
                        [-3, 0, 7, 6, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

     ZR( ['g(257K)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 6, 0, 0 ],
                        [-5, 0, 0, 0, 7, 0, 0 ],
                        [ 0, 0, 0, 6, 0, 0, 0 ],
                        [ 0, 0,-6, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None, # depth
        '1/10*(1575*s^3 - 3762*s^2 + 3000*s - 800)/((9*s - 10)*(7*s - 6)*(7*s - 8)*(6*s - 5)*(3*s - 4)*(s - 1)^2*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['g(257L)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 7, 0, 0, 0 ],
                        [-3, 0, 7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 6, 0, 0 ],
                        [ 0, 0, 0,-6, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ######################################################################
    # 2,4,7
    ######################################################################

    ZR( ['g(247A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 6, 7, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(247B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 6, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 7, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,# toric singularities
        None # toric singularities
        ),

    ZR( ['g(247C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 7, 6, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 7, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None # toric singularities
        ),

    ZR( ['g(247D)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 7, 0, 0 ],
                        [-4, 0, 0, 0, 6, 0, 0 ],
                        [-5, 0, 0, 6, 0, 0, 0 ],
                        [ 0, 0,-6, 0, 0, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None # toric singularities
        ),

    ZR( ['g(247E)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 7, 7, 0, 0 ],
                        [-4, 0, 0, 0, 6, 0, 0 ],
                        [-5, 0, 0, 6, 0, 0, 0 ],
                        [-7, 0,-6, 0, 0, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(247F)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0, 0 ],
                        [-4, 0, 0, 6, 0, 0, 0 ],
                        [-5, 0, 0, 0, 7, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None # toric singularities
        ),

    ZR( ['g(247G)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 7, 0, 0, 0 ],
                        [-4, 0, 0, 6, 0, 0, 0 ],
                        [-5, 0, 0, 0, 7, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(247H)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 7, 6, 0, 0 ],
                        [-4, 0, 0, 6, 0, 0, 0 ],
                        [-5, 0, 0, 0, 7, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [-6, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(247I)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 6, 0, 0 ],
                        [-5, 0, 0, 6, 7, 0, 0 ],
                        [ 0, 0,-6, 0, 0, 0, 0 ],
                        [ 0,-6,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(247J)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 6, 0, 0, 0 ],
                        [-4, 0, 0, 0, 6, 0, 0 ],
                        [-5, 0, 0, 6, 7, 0, 0 ],
                        [-6, 0,-6, 0, 0, 0, 0 ],
                        [ 0,-6,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(247K)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 7, 0, 0, 0 ],
                        [-4, 0, 0, 0, 6, 0, 0 ],
                        [-5, 0, 0, 6, 7, 0, 0 ],
                        [-7, 0,-6, 0, 0, 0, 0 ],
                        [ 0,-6,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(247L)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 6, 7, 0, 0 ],
                        [-4, 0, 6, 0, 0, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(247M)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 6, 0, 0 ],
                        [-4, 0, 6, 7, 0, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(247N)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 7, 0, 0 ],
                        [-4, 0, 6, 7, 0, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(247O)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 6, 7, 0, 0 ],
                        [-4, 0, 6, 7, 0, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(247P)', 'g(147E:0)', 'g(147E:1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0, 0 ],
                        [-4, 0, 6, 0, 7, 0, 0 ],
                        [-5,-6, 0, 7, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(247Q)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 6, 0, 0 ],
                        [-4, 0, 6, 0, 7, 0, 0 ],
                        [-5,-6, 0, 7, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(247R)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 6, 6, 0, 0 ],
                        [-4, 0, 6, 0, 7, 0, 0 ],
                        [-5,-6, 0, 7, 0, 0, 0 ],
                        [-6, 0,-7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ######################################################################
    # 2,4,5,7
    ######################################################################

    ZR( ['g(2457A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 7, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 0, 0, 0 ],
                        [-3, 0, 0, 0, 6, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 7, 0, 0 ],
                        [-3, 0, 0, 0, 6, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457D)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 7, 0, 0 ],
                        [-3, 0, 6, 0, 6, 0, 0 ],
                        [-4,-6, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457E)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 0, 0, 0 ],
                        [-3, 0, 6, 0, 6, 0, 0 ],
                        [-4,-6, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457F)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 7, 0, 0 ],
                        [-3, 0, 6, 0, 0, 0, 0 ],
                        [-4,-6, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457G)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 6, 0, 0 ],
                        [-3, 0, 6, 0, 0, 0, 0 ],
                        [-4,-6, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457H)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 0, 0, 0 ],
                        [-3, 0, 6, 0, 7, 0, 0 ],
                        [-4,-6, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457I)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0, 0 ],
                        [-3, 0, 6, 0, 7, 0, 0 ],
                        [-4,-6, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457J)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 1, 0, 0, 0 ),
                (2,0): ( 0, 0, 0,-1, 0, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 0, 1, 1 ),
                (3,0): ( 0, 0, 0, 0, 0,-1,-1 ),

                (1,2): ( 0, 0, 0, 0, 0, 1, 0 ),
                (2,1): ( 0, 0, 0, 0, 0,-1, 0 ),

                (1,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,1): ( 0, 0, 0, 0, 0, 0,-1 ),
                }
            },
        None,
        None
        ),

    ZR( ['g(2457K)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 6, 0, 0 ],
                        [-3, 0, 6, 0, 7, 0, 0 ],
                        [-4,-6, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457L)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0, 0 ],
                        [-3, 0, 5, 0, 7, 0, 0 ],
                        [-4,-5, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2457M)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 6, 0, 0 ],
                        [-3, 0, 5, 6, 0, 0, 0 ],
                        [-4,-5, 0, 0, 0, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ######################################################################
    # 2,3,5,7
    ######################################################################

    ZR( ['g(2357A)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 0, 1, 0, 0 ),
                (2,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 1, 0 ),
                (4,0): ( 0, 0, 0, 0, 0,-1, 0 ),

                (1,3): ( 0, 0, 0, 0, 1, 0, 1 ),
                (3,1): ( 0, 0, 0, 0,-1, 0,-1 ),

                (2,3): ( 0, 0, 0, 0, 0, 1, 0 ),
                (3,2): ( 0, 0, 0, 0, 0,-1, 0 ),

                },
            },
        None,
        None
        ),

    ZR( ['g(2357B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 7, 6, 0, 0 ],
                        [-3, 0, 0, 5, 0, 0, 0 ],
                        [-5, 0, 0, 6, 0, 0, 0 ],
                        [-7,-5,-6, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2357C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 6, 0, 0 ],
                        [-3, 0, 7, 5, 0, 0, 0 ],
                        [-5,-7, 0, 6, 0, 0, 0 ],
                        [ 0,-5,-6, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(2357D)', 'g(1357S:1)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 0, 1, 0, 0 ),
                (2,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 1, 0 ),
                (4,0): ( 0, 0, 0, 0, 0,-1, 0 ),

                (1,2): ( 0, 0, 0, 0, 0, 0, 1 ),
                (2,1): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,3): ( 0, 0, 0, 0, 1, 0, 1 ),
                (3,1): ( 0, 0, 0, 0,-1, 0,-1 ),

                (2,3): ( 0, 0, 0, 0, 0, 1, 0 ),
                (3,2): ( 0, 0, 0, 0, 0,-1, 0 ),
                },
            },
        None,
        None
        ),


    ######################################################################
    # 2,3,4,5,7
    ######################################################################
    ZR( ['g(23457A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 0, 0 ],
                        [-3, 0, 7, 0, 0, 0, 0 ],
                        [-4,-7, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(23457B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0, 0 ],
                        [-3, 0, 7, 0, 6, 0, 0 ],
                        [-4,-7, 0,-6, 0, 0, 0 ],
                        [-5, 0, 6, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(23457C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 0, 0 ],
                        [-3, 0, 0, 0, 7, 0, 0 ],
                        [-4, 0, 0,-7, 0, 0, 0 ],
                        [-5, 0, 7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(23457D)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 0, 0 ],
                        [-3, 0, 6, 0, 7, 0, 0 ],
                        [-4,-6, 0,-7, 0, 0, 0 ],
                        [-5, 0, 7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(23457E)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 1, 0, 0, 0 ),
                (2,0): ( 0, 0, 0,-1, 0, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 1, 0, 0 ),
                (3,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 1, 0 ),
                (4,0): ( 0, 0, 0, 0, 0,-1, 0 ),

                (1,2): ( 0, 0, 0, 0, 1, 0, 1 ),
                (2,1): ( 0, 0, 0, 0,-1, 0,-1 ),

                (1,3): ( 0, 0, 0, 0, 0, 1, 0 ),
                (3,1): ( 0, 0, 0, 0, 0,-1, 0 ),
                },
            },
        None,
        None
        ),

    ZR( ['g(23457F)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 1, 0, 0, 0 ),
                (2,0): ( 0, 0, 0,-1, 0, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 1, 0, 0 ),
                (3,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (1,2): ( 0, 0, 0, 0, 1, 0, 1 ),
                (2,1): ( 0, 0, 0, 0,-1, 0,-1 ),

                (1,4): ( 0, 0, 0, 0, 0, 1, 0 ),
                (4,1): ( 0, 0, 0, 0, 0,-1, 0 ),

                (2,3): ( 0, 0, 0, 0, 0,-1, 0 ),
                (3,2): ( 0, 0, 0, 0, 0, 1, 0 ),
                },
            },
        None,
        None
        ),

    ZR( ['g(23457G)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 0, 0 ],
                        [-3, 0, 5, 6, 7, 0, 0 ],
                        [-4,-5, 0,-7, 0, 0, 0 ],
                        [-5,-6, 7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ######################################################################
    # 1,7
    ######################################################################
    ZR( ['g(17)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 7, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 7, 0 ],
                        [ 0, 0, 0, 0,-7, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
                        
            },
        None, # exceeded bound
        '1/((7*s - 6)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*(s - 5)*s)',
        ),

    ######################################################################
    # 1,5,7
    ######################################################################

    ZR( ['g(157)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 7, 0, 0, 0, 0 ],
                        [-3, 0, 0, 7, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 7, 0 ],
                        [ 0, 0, 0, 0,-7, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None, # depth
        '1/((7*s - 6)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)',
        ),

    ######################################################################
    # 1,4,7
    ######################################################################

    ZR( ['g(147A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 7, 0 ],
                        [-4, 0, 0, 0, 7, 0, 0 ],
                        [-5, 0, 0, 7, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None, # depth
        '2/((7*s - 6)*(5*s - 8)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['g(147B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 7, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 7, 0 ],
                        [-5, 0, 0, 0, 7, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        '2/((7*s - 6)*(5*s - 8)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['g(147C)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 0, 1, 0, 0, 0 ),
                (1,0): ( 0, 0, 0,-1, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 0, 0,-1, 0 ),
                (2,0): ( 0, 0, 0, 0, 0, 1, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 0,-1 ),
                (4,0): ( 0, 0, 0, 0, 0, 0, 1 ),

                (1,2): ( 0, 0, 0, 0, 1, 0, 0 ),
                (2,1): ( 0, 0, 0, 0,-1, 0, 0 ),

                (1,5): ( 0, 0, 0, 0, 0, 0, Integer(1)/Integer(2) ),
                (5,1): ( 0, 0, 0, 0, 0, 0,-Integer(1)/Integer(2) ),

                (2,3): ( 0, 0, 0, 0, 0, 0, Integer(1)/Integer(2) ),
                (3,2): ( 0, 0, 0, 0, 0, 0,-Integer(1)/Integer(2) ),
                },
            },
        None,
        None
        ),

    ZR( ['g(147D)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 0, 1, 0, 0, 0 ),
                (1,0): ( 0, 0, 0,-1, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 0, 0,-1, 0 ),
                (2,0): ( 0, 0, 0, 0, 0, 1, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 0,-1 ),
                (4,0): ( 0, 0, 0, 0, 0, 0, 1 ),

                (1,2): ( 0, 0, 0, 0, 1, 0, 0 ),
                (2,1): ( 0, 0, 0, 0,-1, 0, 0 ),

                (1,5): ( 0, 0, 0, 0, 0, 0, Integer(1)/Integer(2)),
                (5,1): ( 0, 0, 0, 0, 0, 0,-Integer(1)/Integer(2)),

                (2,3): ( 0, 0, 0, 0, 0, 0, Integer(1)/Integer(2)),
                (3,2): ( 0, 0, 0, 0, 0, 0,-Integer(1)/Integer(2)),

                (2,5): ( 0, 0, 0, 0, 0, 0, 1 ),
                (5,2): ( 0, 0, 0, 0, 0, 0,-1 ),
                },
            },
        None,
        None
        ),

    # The following two algebras are removed since they are duplicates;
    # see Magnin's paper.

    # ZR( ['g(147E:0)'],
    #     {
    #         'simple_basis': True,
    #         'matrix': [ [ 0, 4,-6, 0,-7, 0, 0 ],
    #                     [-4, 0, 5, 0, 0, 0, 0 ],
    #                     [ 6,-5, 0, 7, 0, 0, 0 ],
    #                     [ 0, 0,-7, 0, 0, 0, 0 ],
    #                     [ 7, 0, 0, 0, 0, 0, 0 ],
    #                     [ 0, 0, 0, 0, 0, 0, 0 ],
    #                     [ 0, 0, 0, 0, 0, 0, 0 ] ],
    #         },
    #     None,
    #     None
    #     ),


    # ZR( ['g(147E:1)'],
    #     {
    #         'simple_basis': True,
    #         'matrix': [ [ 0, 4,-6, 0,-7, 0, 0 ],
    #                     [-4, 0, 5, 0, 0, 7, 0 ],
    #                     [ 6,-5, 0, 0, 0, 0, 0 ],
    #                     [ 0, 0, 0, 0, 0, 0, 0 ],
    #                     [ 7, 0, 0, 0, 0, 0, 0 ],
    #                     [ 0,-7, 0, 0, 0, 0, 0 ],
    #                     [ 0, 0, 0, 0, 0, 0, 0 ] ],
    #         },
    #     None,
    #     None
    #     ),

    ######################################################################
    # 1,4,5,7
    ######################################################################

    ZR( ['g(1457A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 7, 0 ],
                        [ 0, 0, 0, 0,-7, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1457B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 0, 0, 0 ],
                        [-3, 0, 7, 0, 0, 0, 0 ],
                        [-4,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 7, 0 ],
                        [ 0, 0, 0, 0,-7, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ######################################################################
    # 1,3,7
    ######################################################################

    ZR( ['g(137A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 7, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 6, 0, 7, 0 ],
                        [ 0, 0,-6, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(137B)'],
        # NOTE: This is the corrected version of this algebra;
        # see Magnin's paper cited below.
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 7, 0, 0 ],
                        [-5, 0, 0, 7, 0, 0, 0 ],
                        [ 0, 0, 0, 6, 0, 7, 0 ],
                        [ 0,-7,-6, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None, # too big for Singular
        '2/15*(1080*s^4 - 2889*s^3 + 2858*s^2 - 1240*s + 200)/((8*s - 5)*(7*s - 4)*(6*s - 5)*(3*s - 2)*(3*s - 4)^2*(s - 1)^2*(s - 2)*(s - 3)*s)', 
        ),

    ZR( ['g(137C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 6, 0, 7, 0 ],
                        [-5, 0, 6, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0,-7, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 7, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None, # GB computations + probably depth
        '1/45*(76032*s^5 - 302247*s^4 + 471582*s^3 - 360697*s^2 + 135100*s - 19800)/((16*s - 11)*(11*s - 10)*(7*s - 4)*(6*s - 5)*(3*s - 2)*(3*s - 4)*(2*s - 3)*(s - 1)^2*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['g(137D)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 6, 0, 7, 0 ],
                        [-5, 0, 6, 7, 0, 0, 0 ],
                        [ 0,-6, 0, 0,-7, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 7, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        '1/45*(76032*s^5 - 302247*s^4 + 471582*s^3 - 360697*s^2 + 135100*s - 19800)/((16*s - 11)*(11*s - 10)*(7*s - 4)*(6*s - 5)*(3*s - 2)*(3*s - 4)*(2*s - 3)*(s - 1)^2*(s - 2)*(s - 3)*s)',
        ),

    ######################################################################
    # 1,3,5,7
    ######################################################################
    ZR( ['g(1357A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 7, 0, 0 ],
                        [-3, 0, 0, 5, 0, 7, 0 ],
                        [-5, 0, 0, 7, 0, 0, 0 ],
                        [ 0,-5,-7, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(1357B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 7, 0, 0 ],
                        [-3, 0, 0, 5, 0, 0, 0 ],
                        [-5, 0, 0, 7, 0, 0, 0 ],
                        [ 0,-5,-7, 0, 0, 7, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 7, 0, 0 ],
                        [-3, 0, 7, 5, 0, 0, 0 ],
                        [-5,-7, 0, 7, 0, 0, 0 ],
                        [ 0,-5,-7, 0, 0, 7, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357D)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 7, 0, 0 ],
                        [-3, 0, 0, 0, 0, 7, 0 ],
                        [-5, 0, 0,-7, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357E)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 7, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 7, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357F)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 7, 0, 0 ],
                        [-3, 0, 7, 0, 0, 0, 0 ],
                        [-5,-7, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 7, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357G)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 7, 0, 0 ],
                        [-3, 0, 0, 6, 0, 7, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357H)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 7, 7, 0 ],
                        [-3, 0, 0, 6, 0, 7, 0 ],
                        [-5, 0, 0, 7, 0, 0, 0 ],
                        [ 0,-6,-7, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [-7,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357I)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 7, 0, 0 ],
                        [-3, 0, 0, 6, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 7, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357J)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 7, 0, 0 ],
                        [-3, 0, 7, 6, 0, 0, 0 ],
                        [-5,-7, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 7, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357K)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 0, 1, 0, 0 ),
                (2,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 0, 1, 0 ),
                (3,0): ( 0, 0, 0, 0, 0,-1, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,3): ( 0, 0, 0, 0, 1, 0, 0 ),
                (3,1): ( 0, 0, 0, 0,-1, 0, 0 ),

                (1,5): ( 0, 0, 0, 0, 0, 0, Integer(1)/Integer(2) ),
                (5,1): ( 0, 0, 0, 0, 0, 0,-Integer(1)/Integer(2) ),
                
                (2,3): ( 0, 0, 0, 0, 0, 0, Integer(1)/Integer(2) ),
                (3,2): ( 0, 0, 0, 0, 0, 0,-Integer(1)/Integer(2) ),
                },
            },
        None,
        None
        ),


    ZR( ['g(1357L)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 0, 1, 0, 0 ),
                (2,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 0, 1, 0 ),
                (3,0): ( 0, 0, 0, 0, 0,-1, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,2): ( 0, 0, 0, 0, 0, 0, 1 ),
                (2,1): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,3): ( 0, 0, 0, 0, 1, 0, 0 ),
                (3,1): ( 0, 0, 0, 0,-1, 0, 0 ),

                (1,5): ( 0, 0, 0, 0, 0, 0, Integer(1)/Integer(2) ),
                (5,1): ( 0, 0, 0, 0, 0, 0,-Integer(1)/Integer(2) ),
                
                (2,3): ( 0, 0, 0, 0, 0, 0, Integer(1)/Integer(2) ),
                (3,2): ( 0, 0, 0, 0, 0, 0,-Integer(1)/Integer(2) ),
                },
            },
        None,
        None
        ),


    ZR( ['g(1357M:0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 7, 0, 0 ],
                        [-3, 0, 0, 5, 0, 0, 0 ],
                        [-5, 0, 0, 7, 0, 0, 0 ],
                        [-6,-5,-7, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357M:1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 7, 0, 0 ],
                        [-3, 0, 0, 5, 0, 7, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-6,-5, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357N:0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 7, 0, 0 ],
                        [-3, 0, 0, 5, 0, 0, 0 ],
                        [-5, 0, 0, 7, 0, 0, 0 ],
                        [-6,-5,-7, 0, 0, 7, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357N:1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 7, 0, 0 ],
                        [-3, 0, 7, 5, 0, 0, 0 ],
                        [-5,-7, 0, 7, 0, 0, 0 ],
                        [-6,-5,-7, 0, 0, 7, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357N:-1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 7, 0, 0 ],
                        [-3, 0,-7, 5, 0, 0, 0 ],
                        [-5, 7, 0, 7, 0, 0, 0 ],
                        [-6,-5,-7, 0, 0, 7, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(1357O)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 0, 7, 0 ],
                        [-3, 0, 6, 0, 7, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(1357P)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 7, 0, 0 ],
                        [-3, 0, 6, 0, 0, 7, 0 ],
                        [-5,-6, 0,-7, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(1357Q)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 1, 0, 0, 0 ),
                (1,0): ( 0, 0,-1,-1, 0, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 1, 0, 0 ),
                (3,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,2): ( 0, 0, 0, 0, 0, 1, 0 ),
                (2,1): ( 0, 0, 0, 0, 0,-1, 0 ),

                (1,5): ( 0, 0, 0, 0, 0, 0, 1 ),
                (5,1): ( 0, 0, 0, 0, 0, 0,-1 ),
                },
            },
        None,
        None
        ),

    ZR( ['g(1357R)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 1, 0, 0, 0 ),
                (1,0): ( 0, 0,-1,-1, 0, 0, 0 ),
                
                (0,3): ( 0, 0, 0, 0, 1, 0, 0 ),
                (3,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,5): ( 0, 0, 0, 0, 0, 0, 1 ),
                (5,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,2): ( 0, 0, 0, 0, 0, 1, 0 ),
                (2,1): ( 0, 0, 0, 0, 0,-1, 0 ),

                (1,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,1): ( 0, 0, 0, 0, 0, 0,-1 ),

                (2,3): ( 0, 0, 0, 0, 0, 0,-1 ),
                (3,2): ( 0, 0, 0, 0, 0, 0, 1 ),
                },
            },
        None,
        None
        ),

    ZR( ['g(1357S:0)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 1, 0, 0, 0 ),
                (1,0): ( 0, 0,-1,-1, 0, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 1, 0, 0 ),
                (3,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (0,5): ( 0, 0, 0, 0, 0, 0, 1 ),
                (5,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,2): ( 0, 0, 0, 0, 0, 1, -0 ),
                (2,1): ( 0, 0, 0, 0, 0,-1, -0 ),

                (1,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,1): ( 0, 0, 0, 0, 0, 0,-1 ),

                (2,3): ( 0, 0, 0, 0, 0, 0,-1 ),
                (3,2): ( 0, 0, 0, 0, 0, 0, 1 ),
                },
            },
        None,
        None
        ),

    # The following algebra has been removed since it is a duplicate;
    # see Magnin's paper.
    # ZR( ['g(1357S:1)'],
    #     {
    #         'rank': 7,
    #         'table': {
    #             (0,1): ( 0, 0, 1, 1, 0, 0, 0 ),
    #             (1,0): ( 0, 0,-1,-1, 0, 0, 0 ),

    #             (0,3): ( 0, 0, 0, 0, 1, 0, 0 ),
    #             (3,0): ( 0, 0, 0, 0,-1, 0, 0 ),

    #             (0,4): ( 0, 0, 0, 0, 0, 0, 1 ),
    #             (4,0): ( 0, 0, 0, 0, 0, 0,-1 ),

    #             (0,5): ( 0, 0, 0, 0, 0, 0, 1 ),
    #             (5,0): ( 0, 0, 0, 0, 0, 0,-1 ),

    #             (1,2): ( 0, 0, 0, 0, 0, 1, -0 ),
    #             (2,1): ( 0, 0, 0, 0, 0,-1, -0 ),

    #             (1,4): ( 0, 0, 0, 0, 0, 0, 1 ),
    #             (4,1): ( 0, 0, 0, 0, 0, 0,-1 ),

    #             (1,5): ( 0, 0, 0, 0, 0, 0, 1 ),
    #             (5,1): ( 0, 0, 0, 0, 0, 0,-1 ),

    #             (2,3): ( 0, 0, 0, 0, 0, 0,-1 ),
    #             (3,2): ( 0, 0, 0, 0, 0, 0, 1 ),
    #             },
    #         },
    #     None,
    #     None
    #     ),

    ZR( ['g(1357S:-1)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 1, 0, 0, 0 ),
                (1,0): ( 0, 0,-1,-1, 0, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 1, 0, 0 ),
                (3,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (0,5): ( 0, 0, 0, 0, 0, 0, 1 ),
                (5,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,2): ( 0, 0, 0, 0, 0, 1, -0 ),
                (2,1): ( 0, 0, 0, 0, 0,-1, -0 ),

                (1,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,1): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,5): ( 0, 0, 0, 0, 0, 0,-1 ),
                (5,1): ( 0, 0, 0, 0, 0, 0, 1 ),

                (2,3): ( 0, 0, 0, 0, 0, 0,-1 ),
                (3,2): ( 0, 0, 0, 0, 0, 0, 1 ),
                },
            },
        None,
        None
        ),

    ######################################################################
    # 1,3,4,5,7
    ######################################################################
    ZR( ['g(13457A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 7, 0, 0 ],
                        [-3, 0, 0, 0, 0, 7, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(13457B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 7, 0, 0 ],
                        [-3, 0, 7, 0, 0, 7, 0 ],
                        [-4,-7, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(13457C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 7, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(13457D)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 7, 0, 0 ],
                        [-3, 0, 5, 7, 0, 7, 0 ],
                        [-4,-5, 0, 0, 0, 0, 0 ],
                        [-5,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(13457E)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 7, 0 ],
                        [-3, 0, 5, 0, 7, 0, 0 ],
                        [-4,-5, 0,-7, 0, 0, 0 ],
                        [-5, 0, 7, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['g(13457F)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 7, 0, 0 ],
                        [-3, 0, 6, 0, 0, 7, 0 ],
                        [-4,-6, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(13457G)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 7, 0 ],
                        [-3, 0, 6, 7, 7, 0, 0 ],
                        [-4,-6, 0,-7, 0, 0, 0 ],
                        [-5,-7, 7, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    #
    # This algebra is in Seeley's list but it's not a Lie algebra,
    # see Magnin's paper "Determination of 7-Dimensional Indecomposable
    # Nilpotent Complex Lie Algebras by Adjoining a Derivation to
    # 6-Dimensional Lie Algebras" 
    #
    # ZR( ['g(13457H)'],
    #     {
    #         'simple_basis': True,
    #         'matrix': [ [ 0, 3, 4, 5, 0, 0, 0 ],
    #                     [-3, 0, 6, 7, 7, 7, 0 ],
    #                     [-4,-6, 0,-7, 0, 0, 0 ],
    #                     [-5,-7, 7, 0, 0, 0, 0 ],
    #                     [ 0,-7, 0, 0, 0, 0, 0 ],
    #                     [ 0,-7, 0, 0, 0, 0, 0 ],
    #                     [ 0, 0, 0, 0, 0, 0, 0 ] ],
    #         },
    #     None,
    #     None
    #     ),

    ZR( ['g(13457I)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 7, 0 ],
                        [-3, 0, 6, 7, 7, 7, 0 ],
                        [-4,-6, 0,-7, 0, 0, 0 ],
                        [-5,-7, 7, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [-7,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ######################################################################
    # 1,2,4,5,7
    ######################################################################
    ZR( ['g(12457A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 7, 0 ],
                        [-3, 0, 0, 0, 6, 0, 0 ],
                        [-4, 0, 0, 0, 7, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-6,-7, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457B)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 1, 0, 0, 0 ),
                (2,0): ( 0, 0, 0,-1, 0, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 0, 1, 0 ),
                (3,0): ( 0, 0, 0, 0, 0,-1, 0 ),

                (0,5): ( 0, 0, 0, 0, 0, 0, 1 ),
                (5,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,4): ( 0, 0, 0, 0, 0, 1, 1 ),
                (4,1): ( 0, 0, 0, 0, 0,-1,-1 ),

                (2,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,2): ( 0, 0, 0, 0, 0, 0,-1 ),
                },
            },
        None,
        None
        ),

    ZR( ['g(12457C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0, 0 ],
                        [-3, 0, 0, 0, 6, 7, 0 ],
                        [-4, 0, 0,-7, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457D)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 7, 0, 0 ],
                        [-3, 0, 0, 0, 6, 7, 0 ],
                        [-4, 0, 0,-7, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457E)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 7, 0 ],
                        [-3, 0, 6, 7, 6, 0, 0 ],
                        [-4,-6, 0, 0, 7, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [ 0,-6,-7, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457F)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0, 0 ],
                        [-3, 0, 6, 0, 6, 7, 0 ],
                        [-4,-6, 0,-7, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457G)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 7, 0, 0 ],
                        [-3, 0, 6, 0, 6, 7, 0 ],
                        [-4,-6, 0,-7, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0 ],
                        [-7,-6, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457H)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 6, 7, 0 ],
                        [-3, 0, 5, 6, 0, 0, 0 ],
                        [-4,-5, 0, 7, 0, 0, 0 ],
                        [ 0,-6,-7, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457I)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 6, 7, 0 ],
                        [-3, 0, 5, 6, 7, 0, 0 ],
                        [-4,-5, 0, 7, 0, 0, 0 ],
                        [ 0,-6,-7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457J)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 6, 7, 0 ],
                        [-3, 0, 5, 6, 7, 0, 0 ],
                        [-4,-5, 0, 7, 0, 0, 0 ],
                        [-7,-6,-7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457K)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 6, 7, 0 ],
                        [-3, 0, 5, 6, 0, 0, 0 ],
                        [-4,-5, 0, 7, 0, 0, 0 ],
                        [-7,-6,-7, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457L)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 6, 7, 0 ],
                        [-3, 0, 5, 6, 0, 7, 0 ],
                        [-4,-5, 0, 7, 0, 0, 0 ],
                        [ 0,-6,-7, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457M)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 6, 7, 0 ],
                        [-3, 0, 5, 6, 0, 7, 0 ],
                        [-4,-5, 0, 7,-7, 0, 0 ],
                        [-7,-6,-7, 0, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0 ],
                        [-7,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457N:1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 6, 7, 0 ],
                        [-3, 0, 5, 6, 7, 7, 0 ],
                        [-4,-5, 0, 7,-7, 0, 0 ],
                        [-7,-6,-7, 0, 0, 0, 0 ],
                        [-6,-7, 7, 0, 0, 0, 0 ],
                        [-7,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12457N:-1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 7, 6, 7, 0 ],
                        [-3, 0, 5, 6,-7, 7, 0 ],
                        [-4,-5, 0, 7,-7, 0, 0 ],
                        [-7,-6,-7, 0, 0, 0, 0 ],
                        [-6, 7, 7, 0, 0, 0, 0 ],
                        [-7,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ######################################################################
    # 1,2,3,5,7
    ######################################################################
    ZR( ['g(12357A)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 6, 7, 0 ],
                        [-3, 0, 0, 5, 0, 0, 0 ],
                        [-5, 0, 0, 6, 0, 0, 0 ],
                        [ 0,-5,-6, 0,-7, 0, 0 ],
                        [-6, 0, 0, 7, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(12357B)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 0, 1, 0, 0 ),
                (2,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 1, 0 ),
                (4,0): ( 0, 0, 0, 0, 0,-1, 0 ),

                (0,5): ( 0, 0, 0, 0, 0, 0, 1 ),
                (5,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,3): ( 0, 0, 0, 0, 1, 0, 1 ),
                (3,1): ( 0, 0, 0, 0,-1, 0,-1 ),

                (2,3): ( 0, 0, 0, 0, 0, 1, 0 ),
                (3,2): ( 0, 0, 0, 0, 0,-1, 0 ),

                (3,4): ( 0, 0, 0, 0, 0, 0,-1 ),
                (4,3): ( 0, 0, 0, 0, 0, 0, 1 ),
                },
            },
        None,
        None
        ),

    ZR( ['g(12357C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 6, 7, 0 ],
                        [-3, 0, 7, 5, 0, 0, 0 ],
                        [-5,-7, 0, 6, 0, 0, 0 ],
                        [ 0,-5,-6, 0,-7, 0, 0 ],
                        [-6, 0, 0, 7, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ######################################################################
    # 1,2,3,4,5,7
    ######################################################################
    ZR( ['g(123457A)', 'M6'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 7, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(123457B)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 7, 0 ],
                        [-3, 0, 7, 0, 0, 0, 0 ],
                        [-4,-7, 0, 0, 0, 0, 0 ],
                        [-5, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(123457C)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 7, 0 ],
                        [-3, 0, 0, 0, 7, 0, 0 ],
                        [-4, 0, 0,-7, 0, 0, 0 ],
                        [-5, 0, 7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(123457D)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 7, 0 ],
                        [-3, 0, 6, 7, 0, 0, 0 ],
                        [-4,-6, 0, 0, 0, 0, 0 ],
                        [-5,-7, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(123457E)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 1, 0, 0, 0 ),
                (2,0): ( 0, 0, 0,-1, 0, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 1, 0, 0 ),
                (3,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 1, 0 ),
                (4,0): ( 0, 0, 0, 0, 0,-1, 0 ),

                (0,5): ( 0, 0, 0, 0, 0, 0, 1 ),
                (5,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,2): ( 0, 0, 0, 0, 0, 1, 1 ),
                (2,1): ( 0, 0, 0, 0, 0,-1,-1 ),

                (1,3): ( 0, 0, 0, 0, 0, 0, 1 ),
                (3,1): ( 0, 0, 0, 0, 0, 0,-1 ),
                },
            },
        None,
        None
        ),

    ZR( ['g(123457F)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 7, 0 ],
                        [-3, 0, 6, 7, 7, 0, 0 ],
                        [-4,-6, 0,-7, 0, 0, 0 ],
                        [-5,-7, 7, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(123457G)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 7, 0 ],
                        [-3, 0, 5, 6, 7, 0, 0 ],
                        [-4,-5, 0, 0, 0, 0, 0 ],
                        [-5,-6, 0, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['g(123457H)'],
        {
            'rank': 7,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 1, 0, 0, 0 ),
                (2,0): ( 0, 0, 0,-1, 0, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 1, 0, 0 ),
                (3,0): ( 0, 0, 0, 0,-1, 0, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 1, 0 ),
                (4,0): ( 0, 0, 0, 0, 0,-1, 0 ),

                (0,5): ( 0, 0, 0, 0, 0, 0, 1 ),
                (5,0): ( 0, 0, 0, 0, 0, 0,-1 ),

                (1,2): ( 0, 0, 0, 0, 1, 0, 1 ),
                (2,1): ( 0, 0, 0, 0,-1, 0,-1 ),

                (1,3): ( 0, 0, 0, 0, 0, 1, 0 ),
                (3,1): ( 0, 0, 0, 0, 0,-1, 0 ),

                (1,4): ( 0, 0, 0, 0, 0, 0, 1 ),
                (4,1): ( 0, 0, 0, 0, 0, 0,-1 ),
                },
            },
        None,
        None
        ),

    ZR( ['g(123457I:0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 7, 0 ],
                        [-3, 0, 5, 6, 0, 0, 0 ],
                        [-4,-5, 0, 7, 0, 0, 0 ],
                        [-5,-6,-7, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ######################################################################
    # NILPOTENT LIE ALGEBRAS OF DIM > 7
    ######################################################################

    ZR( ['N_1^(8,2)'],
         {
             'rank': 8,
             'table': {
                 (0,1): ( 0, 0, 0, 0, 0, 0, 1, 0 ),
                 (1,0): ( 0, 0, 0, 0, 0, 0,-1, 0 ),

                 (2,3): ( 0, 0, 0, 0, 0, 0, 0, 1 ),
                 (3,2): ( 0, 0, 0, 0, 0, 0, 0,-1 ),

                 (4,5): ( 0, 0, 0, 0, 0, 0, 1, 1 ),
                 (5,4): ( 0, 0, 0, 0, 0, 0,-1,-1 ),
                 },
             'descr': "See Ren, Zhu: `Classification of 2-step nilpotent Lie algebras of dimension 8 with 2-dimensional center (2011)",
             },
         None,
         None,
         ),

    ZR( ['N_2^(8,2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 7, 8, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0, 0 ],
                        [-8, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 7, 8, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0,-8, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Ren, Zhu: `Classification of 2-step nilpotent Lie algebras of dimension 8 with 2-dimensional center (2011)",
            },
        None,
        None
        ),

    ZR( ['N_3^(8,2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 7, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 8, 0, 0, 0, 0 ],
                        [ 0, 0,-8, 0, 7, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 8, 0, 0 ],
                        [ 0, 0, 0, 0,-8, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Ren, Zhu: `Classification of 2-step nilpotent Lie algebras of dimension 8 with 2-dimensional center (2011)",
            },
        None,
        None
        ),

    ZR( ['N_4^(8,2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 7, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 7, 0, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 8, 0, 0, 0 ],
                        [ 0, 0, 0,-8, 0, 7, 0, 0 ],
                        [ 0, 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Ren, Zhu: `Classification of 2-step nilpotent Lie algebras of dimension 8 with 2-dimensional center (2011)",
            },
        None,
        None
        ),

    ZR( ['N_5^(8,2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 7, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 8, 0, 0, 0, 0, 0 ],
                        [ 0,-8, 0, 7, 0, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 8, 0, 0, 0 ],
                        [ 0, 0, 0,-8, 0, 7, 0, 0 ],
                        [ 0, 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Ren, Zhu: `Classification of 2-step nilpotent Lie algebras of dimension 8 with 2-dimensional center (2011)",
            },
        None,
        None
        ),

    ZR( ['N_1^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 8, 0, 0, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0, 0 ],
                        [-8,-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 6, 0, 0, 0 ],
                        [ 0, 0, 0,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_2^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 8, 0, 0, 0, 0 ],
                        [ 0, 0,-8, 0, 6, 0, 0, 0 ],
                        [ 0, 0, 0,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_3^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 8, 0, 0, 0, 0 ],
                        [ 0, 0,-8, 0, 7, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_4^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 0, 0,-8, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 8, 0, 0, 0, 0 ],
                        [ 0, 0,-8, 0, 7, 0, 0, 0 ],
                        [ 8, 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_5^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 7, 8, 0, 0, 0, 0 ],
                        [-6, 0, 0, 0, 7, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0, 0 ],
                        [-8, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_6^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 7, 8, 0, 0, 0, 0 ],
                        [-6, 0, 8, 0, 0, 0, 0, 0 ],
                        [-7,-8, 0, 0, 0, 0, 0, 0 ],
                        [-8, 0, 0, 0, 7, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_7^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 7, 0, 8, 0, 0, 0 ],
                        [-6, 0, 0, 8, 0, 0, 0, 0 ],
                        [-7, 0, 0, 6, 0, 0, 0, 0 ],
                        [ 0,-8,-6, 0, 0, 0, 0, 0 ],
                        [-8, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_8^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 7, 8, 0, 0, 0, 0 ],
                        [-6, 0, 8, 0, 7, 0, 0, 0 ],
                        [-7,-8, 0, 0, 0, 0, 0, 0 ],
                        [-8, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_9^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 7, 8, 0, 0, 0, 0 ],
                        [-6, 0, 8, 0, 7, 0, 0, 0 ],
                        [-7,-8, 0, 0, 0, 0, 0, 0 ],
                        [-8, 0, 0, 0, 6, 0, 0, 0 ],
                        [ 0,-7, 0,-6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_10^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 0, 0, 0, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 7, 0, 0, 0, 0 ],
                        [ 0, 0,-7, 0, 8, 0, 0, 0 ],
                        [ 0, 0, 0,-8, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_11^(8,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 0, 0,-7, 0, 0, 0 ],
                        [-6, 0, 7, 0, 0, 0, 0, 0 ],
                        [ 0,-7, 0, 8, 0, 0, 0, 0 ],
                        [ 0, 0,-8, 0, 7, 0, 0, 0 ],
                        [ 7, 0, 0,-7, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_1^(8,4)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0,-8, 0, 0, 0, 0 ],
                        [-5, 0, 6, 0, 0, 0, 0, 0 ],
                        [ 0,-6, 0, 7, 0, 0, 0, 0 ],
                        [ 8, 0,-7, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_2^(8,4)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 6, 8, 0, 0, 0, 0 ],
                        [-5, 0, 7, 0, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0, 0 ],
                        [-8, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['N_3^(8,4)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 6, 8, 0, 0, 0, 0 ],
                        [-5, 0, 7, 6, 0, 0, 0, 0 ],
                        [-6,-7, 0, 0, 0, 0, 0, 0 ],
                        [-8,-6, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            'descr': "See Yan, Deng: `The classification of two step nilpotent complex Lie  algebras of dimension 8' (2013)",
            },
        None,
        None
        ),

    ZR( ['LL72_L1'], 
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 6,   7, 0, 0,   9, 0, 0 ],
                        [-4, 0, 5,   6, 9, 0,   0, 0, 0 ],
                        [-6,-5, 0,   0, 8, 0,   0, 9, 0 ],

                        [-7,-6, 0,   0, 0, 0,   0, 0, 0 ],
                        [ 0,-9,-8,   0, 0, 0,   0, 0, 0 ],
                        [ 0, 0, 0,   0, 0, 0,   0, 0, 0 ],

                        [-9, 0, 0,   0, 0, 0,   0, 0, 0 ],
                        [ 0, 0,-9,   0, 0, 0,   0, 0, 0 ],
                        [ 0, 0, 0,   0, 0, 0,   0, 0, 0 ] ],
            'descr': "See Leger and Luks: `On nilpotent groups of algebra automorphisms' (1972)",
            },
        None,
        None
        ),

    ZR( ['LL72_L2'],
        {
            'rank': 8,
            'table': {
                (0,1): ( 0, 0, 1, 0, 0, 0, 0, 0 ),
                (1,0): ( 0, 0,-1, 0, 0, 0, 0, 0 ),

                (0,2): ( 0, 0, 0, 1, 0, 0, 0, 0 ),
                (2,0): ( 0, 0, 0,-1, 0, 0, 0, 0 ),

                (0,3): ( 0, 0, 0, 0, 1, 0, 0, 0 ),
                (3,0): ( 0, 0, 0, 0,-1, 0, 0, 0 ),

                (0,4): ( 0, 0, 0, 0, 0, 1, 0, 0 ),
                (4,0): ( 0, 0, 0, 0, 0,-1, 0, 0 ),

                (0,5): ( 0, 0, 0, 0, 0, 0, 0, 1 ),
                (5,0): ( 0, 0, 0, 0, 0, 0, 0,-1 ),

                (0,6): ( 0, 0, 0, 0, 0, 0, 0, 1 ),
                (6,0): ( 0, 0, 0, 0, 0, 0, 0,-1 ),

                (1,2): ( 0, 0, 0, 0, 1, 0, 0, 0 ),
                (2,1): ( 0, 0, 0, 0,-1, 0, 0, 0 ),

                (1,3): ( 0, 0, 0, 0, 0, 1, 0, 0 ),
                (3,1): ( 0, 0, 0, 0, 0,-1, 0, 0 ),

                (1,4): ( 0, 0, 0, 0, 0, 0, 1, 0 ),
                (4,1): ( 0, 0, 0, 0, 0, 0,-1, 0 ),

                (1,5): ( 0, 0, 0, 0, 0, 0, 0, 2 ),
                (5,1): ( 0, 0, 0, 0, 0, 0, 0,-2 ),

                (2,3): ( 0, 0, 0, 0, 0, 0,-1, 1 ),
                (3,2): ( 0, 0, 0, 0, 0, 0, 1,-1 ),

                (2,4): ( 0, 0, 0, 0, 0, 0, 0,-1 ),
                (4,2): ( 0, 0, 0, 0, 0, 0, 0, 1 ),
                },
            'descr': "See Leger and Luks: `On nilpotent groups of algebra automorphisms' (1972)",
            },
        None,
        None
        ),

    ZR( ['LL72_L3'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 6, 7, 0, 0, 0, 0, 0 ],
                        [-6, 0, 0, 8, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0,-8, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['L(E)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0,   9, 7, 8,   0, 0, 0 ],
                        [ 0, 0, 0,   7, 9, 0,   0, 0, 0 ],
                        [ 0, 0, 0,   8, 0, 7,   0, 0, 0 ],

                        [-9,-7,-8,   0, 0, 0,   0, 0, 0 ],
                        [-7,-9,-0,   0, 0, 0,   0, 0, 0 ],
                        [-8,-0,-7,   0, 0, 0,   0, 0, 0 ],

                        [ 0, 0, 0,   0, 0, 0,   0, 0, 0 ],
                        [ 0, 0, 0,   0, 0, 0,   0, 0, 0 ],
                        [ 0, 0, 0,   0, 0, 0,   0, 0, 0 ] ],
            'descr': "du Sautoy's elliptic curve example",
            },
        None,
        None
        ),

    ZR( ['h(ZZ[X]/X^3)', 'L(3,2) otimes ZZ[X]/X^3'],
        {
            'simple_basis': True,
            'matrix': [[ 0, 3, 0,   0, 6, 0,   0, 9, 0 ],
                       [-3, 0, 0,  -6, 0, 0,  -9, 0, 0 ],
                       [ 0, 0, 0,   0, 0, 0,   0, 0, 0 ],

                       [ 0, 6, 0,   0, 9, 0,   0, 0, 0 ],
                       [-6, 0, 0,  -9, 0, 0,   0, 0, 0 ],
                       [ 0, 0, 0,   0, 0, 0,   0, 0, 0 ],

                       [ 0, 9, 0,   0, 0, 0,   0, 0, 0 ],
                       [-9, 0, 0,   0, 0, 0,   0, 0, 0 ],
                       [ 0, 0, 0,   0, 0, 0,   0, 0, 0 ]],
            },
        None,
        None
        ),

    ZR( ['h(ZZ[X]/X^4)'],
        {
            'simple_basis': True,
            'matrix': [[ 0,  3, 0,   0, 6, 0,    0,  9, 0,   0,12, 0 ],
                       [-3,  0, 0,  -6, 0, 0,   -9,  0, 0, -12, 0, 0 ],
                       [ 0,  0, 0,   0, 0, 0,    0,  0, 0,   0, 0, 0 ],

                       [ 0,  6, 0,   0,  9, 0,   0, 12, 0,   0, 0, 0 ],
                       [-6,  0, 0,  -9,  0, 0, -12,  0, 0,   0, 0, 0 ],
                       [ 0,  0, 0,   0,  0, 0,   0,  0, 0,   0, 0, 0 ],

                       [ 0,  9, 0,   0, 12, 0,   0,  0, 0,   0, 0, 0 ],
                       [-9,  0, 0, -12,  0, 0,   0,  0, 0,   0, 0, 0 ],
                       [ 0,  0, 0,   0,  0, 0,   0,  0, 0,   0, 0, 0 ],

                       [ 0, 12, 0,   0,  0, 0,   0,  0, 0,   0, 0, 0 ],
                       [-12, 0, 0,   0,  0, 0,   0,  0, 0,   0, 0, 0 ],
                       [ 0,  0, 0,   0,  0, 0,   0,  0, 0,   0, 0, 0 ]],
            },
        None,
        None
        ),


    ZR( ['M3 curlyvee M3'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 7, 0, 0, 0, 0 ],
                        [-3, 0, 0, 0, 0, 0, 0 ],
                        [-7, 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 6, 7, 0 ],
                        [ 0, 0, 0,-6, 0, 0, 0 ],
                        [ 0, 0, 0,-7, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['n(5,ZZ)'],
        {
            'matrix_basis': [E(5,0,1), E(5,1,2), E(5,2,3), E(5,3,4),
                             E(5,0,2), E(5,1,3), E(5,2,4),
                             E(5,0,3), E(5,1,4),
                             E(5,0,4)],
            'product':  'Lie',
            },
        None,
        None
        ),

    ZR( ['n(6,ZZ)'],
        {
            'matrix_basis': [E(6,0,1), E(6,1,2), E(6,2,3), E(6,3,4), E(6,4,5),
                             E(6,0,2), E(6,1,3), E(6,2,4), E(6,3,5),
                             E(6,0,3), E(6,1,4), E(6,2,5),
                             E(6,0,4), E(6,1,5),
                             E(6,0,5), 
                         ],
            'product':  'Lie',
            },
        None,
        None
        ),


    ZR( ['F(3,3)'],
        {
            'rank': 14,
            'table': {#             1  2  3    4  5  6  7  8  9 10 11
                (0,1): (0, 0, 0,   -1, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0),
                (1,0): (0, 0, 0,   +1, 0, 0,   0, 0, 0, 0, 0, 0, 0, 0),
                (0,2): (0, 0, 0,    0,+1, 0,   0, 0, 0, 0, 0, 0, 0, 0),
                (2,0): (0, 0, 0,    0,-1, 0,   0, 0, 0, 0, 0, 0, 0, 0),
                (0,3): (0, 0, 0,    0, 0, 0,  -1, 0, 0, 0, 0, 0, 0, 0),
                (3,0): (0, 0, 0,    0, 0, 0,  +1, 0, 0, 0, 0, 0, 0, 0),
                (0,4): (0, 0, 0,    0, 0, 0,   0,-1, 0, 0, 0, 0, 0, 0),
                (4,0): (0, 0, 0,    0, 0, 0,   0,+1, 0, 0, 0, 0, 0, 0),
                (0,5): (0, 0, 0,    0, 0, 0,   0, 0,-1, 0, 0, 0, 0, 0),
                (5,0): (0, 0, 0,    0, 0, 0,   0, 0,+1, 0, 0, 0, 0, 0),

                (1,2): (0, 0, 0,    0, 0,-1,   0, 0, 0, 0, 0, 0, 0, 0),
                (2,1): (0, 0, 0,    0, 0,+1,   0, 0, 0, 0, 0, 0, 0, 0),
                (1,3): (0, 0, 0,    0, 0, 0,   0, 0, 0,-1, 0, 0, 0, 0),
                (3,1): (0, 0, 0,    0, 0, 0,   0, 0, 0,+1, 0, 0, 0, 0),
                (1,4): (0, 0, 0,    0, 0, 0,   0, 0, 0, 0,-1, 0, 0, 0),
                (4,1): (0, 0, 0,    0, 0, 0,   0, 0, 0, 0,+1, 0, 0, 0),
                (1,5): (0, 0, 0,    0, 0, 0,   0, 0, 0, 0, 0,-1, 0, 0),
                (5,1): (0, 0, 0,    0, 0, 0,   0, 0, 0, 0, 0,+1, 0, 0),

                (2,3): (0, 0, 0,    0, 0, 0,   0, 0,+1, 0,+1, 0, 0, 0),
                (3,2): (0, 0, 0,    0, 0, 0,   0, 0,-1, 0,-1, 0, 0, 0),

                (2,4): (0, 0, 0,    0, 0, 0,   0, 0, 0, 0, 0, 0,-1, 0),
                (4,2): (0, 0, 0,    0, 0, 0,   0, 0, 0, 0, 0, 0,+1, 0),
                
                (2,5): (0, 0, 0,    0, 0, 0,   0, 0, 0, 0, 0, 0, 0,-1),
                (5,2): (0, 0, 0,    0, 0, 0,   0, 0, 0, 0, 0, 0, 0,+1),
            },
        },
        None,
        None
        ),

    ####################################
    # NON-NILPOTENT, SOLUBLE LIE RINGS OF DIMENSION 2
    ####################################
    ZR( ['Lie(2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 2 ],
                        [-2, 0 ] ],
            'descr': 'Non-abelian Lie ring of rank 2',
            },
        '1/((s - 1)*s)',
        '1/2/s^2'
        ),

    ####################################
    # NON-NILPOTENT, SOLUBLE LIE RINGS OF DIMENSION 3
    ####################################

    ZR( ['L^2', 's(2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0,-1 ],
                        [ 0, 0,-2 ],
                        [ 1, 2, 0 ] ],
            },
        '1/((s - 1)*(s - 2)*s)',
        '2/3/((2*s - 1)*s^2)',
        ),
    
    ZR( ['L^3(0)',  'Lie(2) (+) (ZZ,0)', 'tr(2,ZZ)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0,-2 ],
                        [ 0, 0,-2 ],
                        [ 2, 2, 0 ] ],

            },
        '1/2*(2*s - 1)/((s - 1)^3*s)', 
        '1/2/((s - 1)*s^2)',
        ),

    ZR( ['L^3(1)'],
        {
            'rank' : 3,
            'table': { (2,0): (0,1,0), (0,2): (0,-1,0),
                       (2,1): (1,1,0), (1,2): (-1,-1,0)
                       },

            },
        '1/2*(2*s - 1)/((s - 1)^3*s)',
        '1/3/s^3',
        ),

    ZR( ['L^4(1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0,-2 ],
                        [ 0, 0,-1 ],
                        [ 2, 1, 0 ] ],
            },
        '1/2*(2*s - 1)/((s - 1)^3*s)',
        '1/3/s^3'
        ),

    ####################################
    # NON-NILPOTENT, SOLUBLE LIE RINGS OF DIMENSION 4
    ####################################
    ZR( ['M^2', 's(3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0,-1 ],
                        [ 0, 0, 0,-2 ],
                        [ 0, 0, 0,-3 ],
                        [ 1, 2, 3, 0 ] ],
            },
        '1/((s - 1)*(s - 2)*(s - 3)*s)',
        '3/4/((3*s - 2)*(s - 1)*s^2)',
        ),

    ZR( ['M^3(0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0,-1 ],
                        [ 0, 0, 0,-3 ],
                        [ 0, 0, 0,-3 ],
                        [ 1, 3, 3, 0 ] ],
            },
        '(2*s - 1)/((2*s - 3)*(s - 1)^2*(s - 2)*s)',
        '2/3/((2*s - 1)*(s - 1)*s^2)',
        ),

    ZR( ['M^3(-1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0,-1 ],
                        [ 0, 0, 0,-3 ],
                        [ 0, 0, 0,-2 ],
                        [ 1, 3, 2, 0 ] ],
            },
        None, # subalgebras & ideals: singular
        None,
        ),

    ZR( ['M^3(1)'],
        {
            'rank': 4,
            'table': { (3,0):(1,0,0,0), (0,3):(-1,0,0,0),
                       (3,1):(0,0,1,0), (1,3):(0,0,-1,0),
                       (3,2):(0,-1,2,0),(2,3):(0,1,-2,0)
                       },
            },
        None, # subalgebras & ideals: singular
        None
        ),

    ZR( ['M^4', 'Lie(2) (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0 ],
                        [ 0, 0, 0,-3 ],
                        [ 0, 0, 0,-3 ],
                        [ 0, 3, 3, 0 ] ],
            },
        '(2*s - 1)/((2*s - 3)*(s - 1)^2*(s - 2)*s)',
        '1/2/((s - 1)*(s - 2)*s^2)',
        ),

    ZR( ['M^6(0,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0,-2 ],
                        [ 0, 0, 0,-3 ],
                        [ 0, 0, 0,-3 ],
                        [ 2, 3, 3, 0 ] ],
            },
        '1/2*(12*s^2 - 23*s + 9)/((2*s - 3)^3*(s - 1)^2*s)',
        '1/4*(5*s - 2)/((3*s - 2)*(2*s - 1)*(s - 1)*s^2)',
        ),

    ZR ( ['M^6(1,0)'],
         {
            'rank': 4,
            'table': {
                (3,0): (0,1,0,0), (0,3): (0,-1,0,0),
                (3,1): (0,0,1,0), (1,3): (0,0,-1,0),
                (3,2): (1,0,1,0), (2,3): (-1,0,-1,0),
                },
            },
         '1/2*(8*s^2 - 12*s + 3)/((2*s - 3)^2*(s - 1)^3*s)',
         '1/4/s^4',
         ),

    ZR ( ['M^6(0,1)'],
         {
            'rank': 4,
            'table': {
                (3,0): (0,1,0,0), (0,3): (0,-1,0,0),
                (3,1): (0,0,1,0), (1,3): (0,0,-1,0),
                (3,2): (0,1,1,0), (2,3): (0,-1,-1,0),
                },
            },
         '1/2*(8*s^2 - 12*s + 3)/((2*s - 3)^2*(s - 1)^3*s)',
         '1/3/((s - 1)*s^3)',
         ),

    ZR ( ['M^6(1,1)'],
         {
            'rank': 4,
            'table': {
                (3,0): (0,1,0,0), (0,3): (0,-1,0,0),
                (3,1): (0,0,1,0), (1,3): (0,0,-1,0),
                (3,2): (1,1,1,0), (2,3): (-1,-1,-1,0),
                },
            },
         None, # subalgebras: singular
         None, # ideals: can't compute Euler characteristics
         ),

    ZR( ['M^7(1,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0,-2 ],
                        [ 0, 0, 0,-3 ],
                        [ 0, 0, 0,-1 ],
                        [ 2, 3, 1, 0 ] ],
            },
        '1/2*(8*s^2 - 12*s + 3)/((2*s - 3)^2*(s - 1)^3*s)',
        '1/4/s^4',
        ),

    ZR( ['M^7(0,1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0,-2 ],
                        [ 0, 0, 0,-3 ],
                        [ 0, 0, 0,-2 ],
                        [ 2, 3, 2, 0 ] ],
            },
        '1/2*(8*s^2 - 12*s + 3)/((2*s - 3)^2*(s - 1)^3*s)',
        '1/3/((s - 1)*s^3)',
        ),

    ZR( ['M^7(1,1)'],
        {
            'rank': 4,
            'table': {
                (3,0): (0,1,0,0), (0,3): (0,-1,0,0),
                (3,1): (0,0,1,0), (1,3): (0,0,-1,0),
                (3,2): (1,1,0,0), (2,3): (-1,-1,0,0),
                },
            },
        '1/2*(8*s^2 - 12*s + 3)/((2*s - 3)^2*(s - 1)^3*s)',
        '1/4/s^4'
        ),

    ZR( ['M^7(1,2)'],
        {
            'rank': 4,
            'table': {
                (3,0): (0,1,0,0), (0,3): (0,-1,0,0),
                (3,1): (0,0,1,0), (1,3): (0,0,-1,0),
                (3,2): (1,2,0,0), (2,3): (-1,-2,0,0),
                },
            },
        '1/2*(8*s^2 - 12*s + 3)/((2*s - 3)^2*(s - 1)^3*s)',
        '1/4/s^4'
        ),

    ZR( ['M^8', 'Lie(2)^2', 'Lie(2) (+) Lie(2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 2, 0, 0 ],
                        [-2, 0, 0, 0 ],
                        [ 0, 0, 0, 4 ],
                        [ 0, 0,-4, 0 ] ],
            },
        '1/2*(10*s^2 - 7*s + 2)/((5*s - 6)*(s - 1)^4*s)',
        '1/4/((s - 1)*s^3)'
        ),

    ZR( ['M^12'],
        {
            'rank': 4,
            'table': {
                (3,0): (1,0,0,0), (0,3):(-1,0,0,0),
                (3,1): (0,2,0,0), (1,3):(0,-2,0,0),
                (3,2): (0,0,1,0), (2,3):(0,0,-1,0),
                (2,0): (0,1,0,0), (0,2):(0,-1,0,0),
                }
            },
        '1/2*(3*s - 2)/((2*s - 3)*(s - 1)^2*(s - 2)*s)',
        '1/6/((2*s - 1)*s^3)',
        ),

    ZR( ['M^13(0)', 'Lie(2)[eps]', 'Lie(2) otimes ZZ[X]/X^2'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0,-2,-1 ],
                        [ 0, 0, 0,-2 ],
                        [ 2, 0, 0,-1 ],
                        [ 1, 2, 1, 0 ] ],
            },
        '1/2*(9*s - 4)/((3*s - 4)*(2*s - 3)*(s - 1)^2*s)',
        '1/2/((4*s - 1)*(s - 1)*s^2)', 
        ),

    ZR( ['M^13(1)'],
        {
            'rank': 4,
            'table': { (3,0):(1,0,1,0), (0,3):(-1,0,-1,0),
                       (3,1):(0,1,0,0), (1,3):(0,-1,0,0),
                       (3,2):(1,0,0,0), (2,3):(-1,0,0,0),
                       (2,0):(0,1,0,0), (0,2):(0,-1,0,0)
                       },
            },
        '3/4*(63*s^3 - 172*s^2 + 142*s - 32)/((7*s - 9)*(3*s - 4)^2*(s - 1)^3*s)',
        None # ideals: singular
        ),

    ZR( ['M^14(1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0,-2,-3 ],
                        [ 0, 0, 0, 0 ],
                        [ 2, 0, 0,-1 ],
                        [ 3, 0, 1, 0 ] ],
            },
        '3/4*(63*s^3 - 172*s^2 + 142*s - 32)/((7*s - 9)*(3*s - 4)^2*(s - 1)^3*s)',
        '1/3/((4*s - 1)*s^3)', 
        ),

    ####################################
    # NON-NILPOTENT, SOLUBLE LIE RINGS OF DIMENSION >= 5
    ####################################
    ZR( ['Lie(2) (+) (ZZ^3,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 2, 0, 0, 0 ],
                        [-2, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],

            },
        '1/2*(2*s - 1)/((s - 1)^2*(s - 2)^2*(s - 3)*s)',
        '1/2/((s - 1)*(s - 2)*(s - 3)*s^2)', 
        ),

    ZR( ['Lie(2) (+) h(ZZ)', 'Lie(2) (+) H(2)'],
        {
            'simple_basis': True,
            'matrix':  [ [ 0, 2, 0, 0, 0 ],
                         [-2, 0, 0, 0, 0 ],
                         [ 0, 0, 0, 5, 0 ],
                         [ 0, 0,-5, 0, 0 ],
                         [ 0, 0, 0, 0, 0 ] ],
            },
        '1/2*(108*s^4 - 505*s^3 + 848*s^2 - 594*s + 140)/((4*s - 7)*(3*s - 4)*(3*s - 5)*(2*s - 3)*(s - 1)^2*(s - 2)^2*s)',
        '1/6/((s - 1)^2*(s - 2)*s^2)', 
        ),

    ZR( ['Lie(2) (+) h(ZZ) (+) (ZZ,0)', 'Lie(2) (+) H(2) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix':  [ [ 0, 2, 0, 0, 0, 0 ],
                         [-2, 0, 0, 0, 0, 0 ],
                         [ 0, 0, 0, 5, 0, 0 ],
                         [ 0, 0,-5, 0, 0, 0 ],
                         [ 0, 0, 0, 0, 0, 0 ],
                         [ 0, 0, 0, 0, 0, 0 ]],
            },
        '1/24*(108*s^4 - 586*s^3 + 1143*s^2 - 921*s + 240)/((3*s - 5)*(2*s - 5)*(s - 1)^2*(s - 2)^4*(s - 3)*s)',
        '1/2/((3*s - 4)*(s - 1)*(s - 2)*(s - 3)*s^2)', 
        ),

    ZR( ['Lie(2)^2 (+) (ZZ,0)', 'Lie(2) (+) Lie(2) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 2, 0, 0, 0 ],
                        [-2, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 4, 0 ],
                        [ 0, 0,-4, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/3*(360*s^6 - 2064*s^5 + 4732*s^4 - 5524*s^3 + 3481*s^2 - 1173*s + 189)/((5*s - 7)*(3*s - 4)*(2*s - 3)^3*(s - 1)^4*(s - 2)*s)', 
        '1/4/((s - 1)*(s - 2)*s^3)',
        ),

    ZR( ['Lie(2)^3', 'Lie(2) (+) Lie(2) (+) Lie(2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 2, 0, 0, 0, 0 ],
                        [-2, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 4, 0, 0 ],
                        [ 0, 0,-4, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 6 ],
                        [ 0, 0, 0, 0,-6, 0 ] ],
            },
        None, # singular
        '1/8/((s - 1)*(s - 2)*s^4)',
        ),

    ZR( ['(Lie(2) otimes ZZ[X]/X^2) (+) (ZZ,0)', 'Lie(2)[eps] (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 2, 0, 4, 0 ],
                        [-2, 0,-4, 0, 0 ],
                        [ 0, 4, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '(108*s^4 - 481*s^3 + 766*s^2 - 501*s + 105)/((4*s - 7)*(3*s - 4)*(3*s - 5)*(2*s - 3)^2*(s - 1)^2*(s - 2)*s)',
        '1/2/((4*s - 1)*(s - 1)*(s - 2)*s^2)'
        ),

    ZR( ['(Lie(2) otimes ZZ[X]/X^2) (+) (ZZ^2,0)', '(Lie(2) (+) (ZZ,0)) otimes ZZ[X]/X^2', 'Lie(2)[eps] (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 2, 0, 4, 0, 0 ],
                        [-2, 0,-4, 0, 0, 0 ],
                        [ 0, 4, 0, 0, 0, 0 ],
                        [-4, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/12*(216*s^5 - 1558*s^4 + 4329*s^3 - 5680*s^2 + 3377*s - 648)/((4*s - 9)*(3*s - 5)*(2*s - 3)*(s - 1)^2*(s - 2)^4*(s - 3)*s)',
        '1/2/((4*s - 1)*(s - 1)*(s - 2)*(s - 3)*s^2)'
        ),

    ####################################
    # INSOLUBLE LIE RINGS
    ####################################
    ZR( ['sl(2,ZZ)'],
        {
            'table' : [ [[ 0, 0, 0], [0, 0, 1], [ 2, 0, 0]],
                        [[ 0, 0,-1], [ 0, 0, 0], [ 0,-2, 0]],
                        [[-2, 0, 0], [ 0, 2, 0], [ 0, 0, 0]] ],
            'descr' : 'Lie ring sl(2,ZZ) with basis (e,f,h)',
            },
        '1/2*(3*s - 1)/((2*s - 1)*(s - 1)^2*s)',
        '0'
        ),

    ZR( ['sl(3,ZZ)'],
        {
            'rank': 8,
            'matrix_basis': [ E(3,0,1), E(3,0,2), E(3,1,2),
                              E(3,1,0), E(3,2,0), E(3,2,1),
                              E(3,0,0) - E(3,1,1),
                              E(3,1,1) - E(3,2,2) ],
            'product': 'Lie',
            },
        None,
        None
        ),

    ZR( ['gl(2,ZZ)', 'sl(2,ZZ) (+) (ZZ,0)'],
        {
            'table' : [ [[ 0, 0, 0, 0], [ 2, 0, 0, 0], [ 0, 1, 0, 0 ], [ 0, 0, 0, 0]],
                        [[-2, 0, 0, 0], [ 0, 0, 0, 0], [ 0, 0, 2, 0 ], [ 0, 0, 0, 0]],
                        [[ 0,-1, 0, 0], [ 0, 0,-2, 0], [ 0, 0, 0, 0 ], [ 0, 0, 0, 0]],
                        [[ 0, 0, 0, 0], [ 0, 0, 0, 0], [ 0, 0, 0, 0 ], [ 0, 0, 0, 0]] ],
            'descr' : 'Lie ring gl(2,ZZ) with basis (e,h,f,1)',
            },
        '1/6*(27*s - 14)/((6*s - 7)*(s - 1)^3*s)',
        '0'
        ),

    ####################################
    # NILPOTENT ASSOCIATIVE ALGEBRAS
    # These algebras are taken from de Graaf: `Classification of nilpotent
    # associative algebras of small dimension' (arXiv:1009.5339).
    ####################################

    ZR( ['A(3,2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 2, 0, 0 ],
                        [ 0, 0, 0 ],
                        [ 0, 0, 0 ] ],
        },
        '2/((3*s - 4)*(s - 1)*s)',
        '1/2/((s - 1)^2*s)',
        ),

    ZR( ['A(3,3;1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 3, 0, 0 ],
                        [ 0, 3, 0 ],
                        [ 0, 0, 0 ] ],
        },
        '1/4*(3*s - 2)/((s - 1)^3*s)',
        '1/((3*s - 2)*(s - 1)*s)',
        ),

    ZR( ['A(3,4;0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 0 ],
                        [ 0, 3, 0 ],
                        [ 0, 0, 0 ] ],
        },
        '1/4*(3*s - 2)/((s - 1)^3*s)',
        '1/((3*s - 2)*(s - 1)*s)',
        ),

    ZR( ['A(3,6)', 'X*ZZ[X]/X^4'],
        {
            'simple_basis': True,
            'matrix': [ [ 2, 3, 0 ],
                        [ 3, 0, 0 ],
                        [ 0, 0, 0 ] ],
        },
        '1/24*(72*s^2 - 131*s + 60)/((6*s - 5)*(s - 1)^3*s)',
        '1/((3*s - 2)*(2*s - 1)*s)',
        ),

    ZR( ['X*ZZ[X]/X^5', 'A(4,8)'],
        {
            'simple_basis' : True,
            'matrix' : [[2,3,4,0],
                        [3,4,0,0],
                        [4,0,0,0],
                        [0,0,0,0]],
        },
        '1/10*(504662400*s^7 - 4095476640*s^6 + 14242040424*s^5 - 27510984828*s^4 + 31880599078*s^3 - 22163002945*s^2 + 8558212705*s - 1416051000)/((15*s - 19)*(12*s - 13)*(11*s - 13)*(10*s - 9)*(6*s - 7)^2*(5*s - 6)*(4*s - 5)^2*(3*s - 4)*s)',
        '1/((4*s - 3)*(3*s - 2)*(2*s - 1)*s)',
        ),
    
    ####################################
    # 'COMMUTATIVE VERSIONS' OF THE NILPOTENT LIE RINGS OF RANK <= 5 FROM ABOVE
    # (Just flip signs below the diagonal.)
    ####################################
    
    ZR( ['C(3,2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 0 ],
                        [ 3, 0, 0 ],
                        [ 0, 0, 0 ] ],
        },
        '1/4*(3*s - 2)/((s - 1)^3*s)',
        '1/((3*s - 2)*(s - 1)*s)',
        ),

    ZR( ['C(4,2)', 'C(3,2) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 0, 0 ],
                        [ 3, 0, 0, 0 ],
                        [ 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0 ] ],
        },
        '3/((2*s - 3)^2*(s - 2)*s)',
        '1/3/((s - 1)^2*(s - 2)*s)',
        ),

    ZR( ['C(4,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0 ],
                        [ 3, 0, 0, 0 ],
                        [ 4, 0, 0, 0 ],
                        [ 0, 0, 0, 0 ] ],
            },
        '1/6*(78*s^2 - 143*s + 60)/((4*s - 5)*(3*s - 4)*(2*s - 3)*(s - 1)^2*s)', 
        '1/2*(5*s - 2)/((5*s - 3)*(3*s - 2)*(2*s - 1)*(s - 1)*s)',
        ),

    ZR( ['C(5,2)', 'C(3,2) (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 0, 0, 0 ],
                        [ 3, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/4*(3*s - 4)/((s - 1)*(s - 2)^3*(s - 3)*s)',
        '1/((3*s - 4)*(s - 1)*(s - 2)*(s - 3)*s)', 
        ),

    ZR( ['C(5,3)', 'C(4,3) (+) (ZZ,0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 0 ],
                        [ 3, 0, 0, 0, 0 ],
                        [ 4, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/3*(39*s^3 - 156*s^2 + 202*s - 84)/((4*s - 7)*(3*s - 4)*(2*s - 3)*(s - 1)*(s - 2)^3*s)', 
        '1/3*(5*s - 3)/((5*s - 4)*(4*s - 3)*(s - 1)^2*(s - 2)*s)', 
        ),
        
    ZR( ['C(5,4)', 'C(3,2) curlyvee C(3,2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 5, 0 ],
                        [ 0, 0, 5, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '(17*s - 25)/((3*s - 4)*(3*s - 5)^2*(s - 2)*(s - 3)*s)',
        '1/((5*s - 4)*(s - 1)*(s - 2)*(s - 3)*s)', 
        ),

    ZR( ['C(5,5)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 0 ],
                        [ 3, 0, 0, 5, 0 ],
                        [ 5, 0, 0, 0, 0 ],
                        [ 0, 5, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/27*(220320*s^6 - 1842072*s^5 + 6367090*s^4 - 11632541*s^3 + 11830093*s^2 - 6337161*s + 1393119)/((9*s - 13)*(5*s - 9)*(4*s - 7)^2*(3*s - 4)*(2*s - 3)^3*(s - 1)*(s - 2)*s)',
        '1/3/((5*s - 4)*(s - 1)^2*(s - 2)*s)',
        ),

    ZR( ['C(5,6)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0 ],
                        [ 3, 0, 5, 0, 0 ],
                        [ 4, 5, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/1080*(18774882624308439288940279132518088704000000000*s^44 - 1106997307621470729219346840157348811571200000000*s^43 + '
        '31878683359134993453386093039762287221596160000000*s^42 - 597503399110376595952020014515215680222736384000000*s^41 + '
        '8195409977007583430239184741454773931661870540800000*s^40 - 87691875443869633883226632480487791255575528942080000*s^39 + '
        '762011194884112906783317269816707701672309875102976000*s^38 - 5527428238837005466380075580444486264180431848290752000*s^37 + '
        '34142565708844849419113042395174338588616927292444439360*s^36 - 182305590873892729611961637657966416675137058233639681936*s^35 + '
        '851317879986490615140555359914344254118933053187406114152*s^34 - 3508963324032647556898489991858214859423503381325704584592*s^33 + '
        '12861351688642493015766824472664721898105147445983128326350*s^32 - 42173495068104608650323851061860505640908126650747186377257*s^31 + '
        '124333385664606016925267192791367036995559781382803983488950*s^30 - 330900312369971462871638409653999119411022769945323038127283*s^29 + '
        '797656942032503573312215380022630351744828125874331603384832*s^28 - 1746321596277300630583773174703198339879981618808795966750432*s^27 + '
        '3479916108744307647857429322013640996883921305364812904113888*s^26 - 6322550181965188315379317318105718622405977945312769336141444*s^25 + '
        '10487029836351430975814686126943023336202734959209328982414124*s^24 - 15894134200512385459637608959651804629290996355660246932380038*s^23 + '
        '22022785454775521427279915184118924577179036734211845111568220*s^22 - 27901686532248322038711785505709291343594747554407399331714770*s^21 + '
        '32316932734456156856272080922663297411431526472717257261126416*s^20 - 34200690680097683071502427411590021473217738105847805725675732*s^19 + '
        '33040575944015526077194727747624858092896036249562969525807128*s^18 - 29100527158519735850173130491473119213630883752781304736002836*s^17 + '
        '23326107252008475029585552264473007378938257406463509949709078*s^16 - 16979091144420470548965520303023293947943008840858382092045785*s^15 + '
        '11192512262240176020009108071497515214661400103701415930316094*s^14 - 6659212758589232022369131510868555422128291374245325576322995*s^13 + '
        '3561422737227761293607267356401158092461643580872481387695200*s^12 - 1703602935442040450637595691415995503465553273682018665760948*s^11 + '
        '724471763577646067458830621605403555076937221151553416164944*s^10 - 271860078027587073094084225628819728122518925515724529670560*s^9 + '
        '89190890166502561791358080941564008210374047198927974980160*s^8 - 25285789756103532318632442578915082477086600521758413319552*s^7 + '
        '6102115031886688802395830301889426965000587545009633914624*s^6 - 1228757793771554810723855406686458358501666191831693515520*s^5 + '
        '200850387243086610256307176579164869794879678540624788480*s^4 - 25599332671645653530508006311240761958319796607709496320*s^3 + '
        '2386432673349745941610482375996990132562805016271872000*s^2 - 144714535590456003683083457070539861190024919449600000*s + '
        '4283478253285228096192677719570198261789491200000000)/((37*s - 58)*(24*s - 31)*(23*s - 32)*(21*s - 25)*(21*s - 26)*(20*s - 29)*'
        '(17*s - 25)*(17*s - 27)*(15*s - 14)*(15*s - 19)*(15*s - 23)*(14*s - 19)*(13*s - 18)*(13*s - 19)^2*(13*s - 21)*(12*s - 13)*'
        '(12*s - 17)^3*(11*s - 14)^2*(11*s - 16)*(11*s - 17)^2*(10*s - 13)^2*(9*s - 13)^3*(9*s - 14)*(8*s - 13)*(7*s - 9)*(7*s - 10)*'
        '(7*s - 11)*(5*s - 6)*(5*s - 7)^2*(5*s - 8)*(4*s - 5)^2*(3*s - 4)*(2*s - 3)^2*(s - 1)^4*s)',
        '1/2*(5040*s^4 - 10157*s^3 + 7522*s^2 - 2412*s + 280)/((12*s - 7)*(8*s - 5)*(7*s - 4)*(5*s - 2)*(5*s - 3)*(3*s - 2)^2*(s - 1)*s)',
        ),

    ZR( ['C(5,7)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0 ],
                        [ 3, 0, 0, 0, 0 ],
                        [ 4, 0, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/72*(4140*s^5 - 23407*s^4 + 51802*s^3 - 56059*s^2 + 29580*s - 6048)/((7*s - 9)*(5*s - 8)*(4*s - 7)*(s - 1)^4*(s - 2)^2*s)',
        '2*(1386*s^3 - 2023*s^2 + 948*s - 140)*(2*s - 1)/((12*s - 7)*(11*s - 6)*(8*s - 5)*(7*s - 4)*(5*s - 2)*(3*s - 2)^2*(s - 1)*s)',
        ),

    ZR( ['C(5,8)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0 ],
                        [ 4, 0, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
        },
        '1/6*(7*s - 12)/((2*s - 3)*(s - 2)^4*s)',
        '2/((5*s - 6)*(3*s - 4)*(s - 1)*(s - 2)*s)', 
        ),

    ZR( ['C(5,9)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 0 ],
                        [ 3, 0, 5, 0, 0 ],
                        [ 4, 5, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ] ],
            },
        '1/6*(21152086956000*s^15 - 423151417319400*s^14 + 3943558602184740*s^13 - 22711174555480221*s^12 +'
        '90389215618290355*s^11 - 263338127267231139*s^10 + 580160351714746447*s^9 -'
        '984196868957355745*s^8 + 1296189686685135409*s^7 - 1325261049322273033*s^6 +'
        '1043313593653271505*s^5 - 621060214800949438*s^4 + 270600244961186328*s^3 -'
        '81468766171906992*s^2 + 15154535967671520*s - 1312986795120000)/('
        '(13*s - 21)*(11*s - 17)*(11*s - 18)*(9*s - 13)*(7*s - 9)*(7*s - 10)*'
        '(7*s - 11)*(5*s - 6)^2*(5*s - 7)^2*(5*s - 8)*(4*s - 5)^2*(3*s - 4)*'
        '(3*s - 5)*(2*s - 3)^2*(s - 1)*s)',
        '2*(35*s^2 - 42*s + 12)/((7*s - 6)*(5*s - 4)^2*(4*s - 3)*(3*s - 2)*(s - 1)*s)',
        ),

    ZR( ['C(6,2)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 0, 0, 0, 0 ],
                        [ 3, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '(3*s - 5)/((2*s - 5)^2*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)',
        '1/((3*s - 5)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)'
        ),

    ZR( ['C(6,3)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 0, 0 ],
                        [ 3, 0, 0, 0, 0, 0 ],
                        [ 4, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/2*(78*s^3 - 403*s^2 + 673*s - 360)/((4*s - 9)*(3*s - 5)*(3*s - 8)*(2*s - 5)*(s - 1)*(s - 2)^2*(s - 3)*s)',
        '1/20*(5*s - 4)/((3*s - 4)*(s - 1)^3*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['C(6,4)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 0, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 5, 0, 0 ],
                        [ 0, 0, 5, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/9*(17*s^2 - 50*s + 36)/((3*s - 5)*(s - 1)*(s - 2)^3*(s - 3)*(s - 4)*s)',
        '1/5/((s - 1)^2*(s - 2)*(s - 3)*(s - 4)*s)',
        ),

    ZR( ['C(6,5)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 0, 0 ],
                        [ 3, 0, 0, 5, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 0, 5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '(220320*s^7 - 2732208*s^6 + 14434662*s^5 - 42088815*s^4 + 73094297*s^3 - '
        '75532858*s^2 + 42949558*s - 10349856)/((9*s - 16)*(6*s - 11)^3*'
        '(5*s - 12)*(4*s - 9)^2*(3*s - 5)*(s - 1)*(s - 2)^2*(s - 3)*s)',
        '1/5/((3*s - 4)*(s - 1)^2*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['C(6,6)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0 ],
                        [ 3, 0, 5, 0, 0, 0 ],
                        [ 4, 5, 0, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None, # main loop finishes successfully; 25114 cases
        '1/6*(5040*s^4 - 13845*s^3 + 14044*s^2 - 6215*s + 1008)/((9*s - 8)*(8*s - 7)*(7*s - 5)*(5*s - 3)*(5*s - 4)*(4*s - 3)*(s - 1)^2*(s - 2)*s)',
        ),

    ZR( ['C(6,7)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0 ],
                        [ 3, 0, 0, 0, 0, 0 ],
                        [ 4, 0, 0, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '(8280*s^5 - 60918*s^4 + 176591*s^3 - 251555*s^2 + 175380*s - 47520)/((7*s - 12)*(5*s - 11)*(4*s - 5)*(4*s - 9)*(3*s - 4)*(3*s - 8)*(2*s - 3)*(2*s - 5)*(s - 1)*(s - 2)*s)',
        '4*(154*s^3 - 309*s^2 + 201*s - 42)*(3*s - 2)/((11*s - 8)*(9*s - 8)*(8*s - 7)*(7*s - 5)*(5*s - 3)*(4*s - 3)*(s - 1)^2*(s - 2)*s)',
        ),

    ZR( ['C(6,8)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0 ],
                        [ 4, 0, 0, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/2*(14*s^2 - 49*s + 40)/((3*s - 8)*(2*s - 5)^2*(s - 1)*(s - 2)^2*(s - 3)*s)',
        '2/((5*s - 8)*(3*s - 5)*(s - 1)*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['C(6,9)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 0, 0, 0 ],
                        [ 3, 0, 5, 0, 0, 0 ],
                        [ 4, 5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/1260*(20306003477760000*s^17 - 625524424953120000*s^16 + 9057626381698171200*s^15 - '
        '81877365020768680320*s^14 + 517487274818918916200*s^13 - 2426704538598780656300*s^12 + '
        '8742853890932162544230*s^11 - 24717312849476062943161*s^10 + 55512951190411737341021*s^9 - '
        '99606391224611410365833*s^8 + 142776664513046115170549*s^7 - 162556242061128199466175*s^6 + '
        '145192942152832459184559*s^5 - 99595476073203752094327*s^4 + 50663300187748548610161*s^3 - '
        '18007418579791943122524*s^2 + 3992439769890017685120*s - 415668720845285107200)/((14*s - 29)*'
        '(13*s - 27)*(11*s - 21)*(11*s - 24)*(10*s - 19)^2*(9*s - 17)*(8*s - 15)*(7*s - 12)*(6*s - 13)*'
        '(5*s - 9)^2*(5*s - 11)*(4*s - 7)^2*(3*s - 5)*(s - 1)*(s - 2)^5*s)',
        '1/30*(35*s - 24)/((7*s - 8)*(5*s - 6)*(s - 1)^3*(s - 2)*s)'
        ),

    ZR( ['C(6,10)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 6, 0, 0, 0 ],
                        [ 3, 0, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 6, 0 ],
                        [ 0, 0, 0, 6, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/3200*(11657100*s^11 - 213081750*s^10 + 1767158385*s^9 - 8776374414*s^8 + 28999047006*s^7 - '
        '66931292792*s^6 + 110098560983*s^5 - 129060759924*s^4 + 105643535726*s^3 - 57502225520*s^2 + '
        '18728536800*s - 2764800000)/((7*s - 12)^2*(3*s - 5)^4*(2*s - 3)*(s - 1)*(s - 2)^7*(s - 3)*s)',
        '1/2*(7*s - 4)/((7*s - 5)*(3*s - 2)*(3*s - 4)*(s - 1)*(s - 2)*(s - 3)*s)'
        ),

    ZR( ['C(6,11)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0 ],
                        [ 3, 0, 6, 0, 6, 0 ],
                        [ 4, 6, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['C(6,12)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0 ], 
                        [ 3, 0, 0, 0, 6, 0 ],
                        [ 4, 0, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None,
        ),

    ZR( ['C(6,13)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 6, 0 ],
                        [ 3, 0, 0, 5, 0, 0 ],
                        [ 5, 0, 0, 6, 0, 0 ],
                        [ 0, 5, 6, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),


    ZR( ['C(6,14)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0 ],
                        [ 3, 0, 5, 0, 6, 0 ],
                        [ 4, 5, 0,-6, 0, 0 ],
                        [ 5, 0,-6, 0, 0, 0 ],
                        [ 0, 6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,15)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 0 ],
                        [ 3, 0, 5, 6, 0, 0 ],
                        [ 4, 5, 0, 0, 0, 0 ],
                        [ 5, 6, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,16)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 0, 0 ],
                        [ 3, 0, 0, 0, 6, 0 ],
                        [ 4, 0, 0,-6, 0, 0 ],
                        [ 5, 0,-6, 0, 0, 0 ],
                        [ 0, 6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,17)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 0 ],
                        [ 3, 0, 6, 0, 0, 0 ],
                        [ 4, 6, 0, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,18)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 5, 6, 0 ],
                        [ 3, 0, 0, 0, 0, 0 ],
                        [ 4, 0, 0, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,19;0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0 ],
                        [ 4, 0, 0, 6, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 0, 6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/7056*(34720812000*s^13 - 776417367600*s^12 + 8000947962000*s^11 - 50306131945512*s^10 + '
        '215308335028338*s^9 - 662372541794643*s^8 + 1506796330037939*s^7 - 2566173790116152*s^6 + '
        '3271659686744033*s^5 - 3083716474422428*s^4 + 2088512876939882*s^3 - 962398505053265*s^2 + '
        '270419622473616*s - 34988805992448)/((9*s - 16)*(6*s - 11)^3*(5*s - 9)^2*(5*s - 11)*'
        '(4*s - 7)^2*(3*s - 5)*(s - 1)*(s - 2)^7*s)',
        '1/210*(455*s^3 - 1144*s^2 + 942*s - 252)/((5*s - 6)*(4*s - 3)*(3*s - 4)*(s - 1)^4*(s - 2)*s)',
        ),

    ZR( ['C(6,19;1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0 ],
                        [ 4, 0, 0, 6, 0, 0 ],
                        [ 5, 0, 0, 0, 6, 0 ],
                        [ 0, 6, 0, 0, 0, 0 ],
                        [ 0, 0, 6, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,20)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 6, 0 ],
                        [ 4, 0, 0, 6, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 0, 6, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,21;0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0 ],
                        [ 3, 0, 5, 0, 0, 0 ],
                        [ 4, 5, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,21;1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 4, 6, 0, 0 ],
                        [ 3, 0, 5, 0, 6, 0 ],
                        [ 4, 5, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,22;0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 6, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 5, 0, 0 ],
                        [ 0, 0, 5, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/9*(1072*s^4 - 7461*s^3 + 19150*s^2 - 21404*s + 8748)/((5*s - 12)*(4*s - 9)^2*(3*s - 5)*(s - 1)*(s - 2)^3*(s - 3)*s)',
        '1/15*(8*s - 9)/((3*s - 4)*(2*s - 3)*(s - 1)^2*(s - 2)*(s - 3)*s)',
        ),

    ZR( ['C(6,22;1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 5, 6, 0, 0, 0 ],
                        [ 5, 0, 0, 6, 0, 0 ],
                        [ 6, 0, 0, 5, 0, 0 ],
                        [ 0, 6, 5, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,23)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 0, 0 ],
                        [ 3, 0, 0, 5, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 6, 5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/105840*(7656569020412928000*s^18 - 256098362074002201600*s^17 + 4040314741843835120640*s^16 - '
        '39945727303449544185696*s^15 + 277332067303198479861696*s^14 - 1435724448094079306191368*s^13 + '
        '5743279326694476168106524*s^12 - 18150531561351403413007428*s^11 + 45935588752816214780459901*s^10 - '
        '93784671910453883289567231*s^9 + 154816973138568357937419625*s^8 - 206121824684395161912177287*s^7 + '
        '219659481768774041443778011*s^6 - 184797573132869294665988125*s^5 + 120011608724929847752011503*s^4 - '
        '58040088734716993157207665*s^3 + 19679098351732355331011700*s^2 - 4173220890976256256456000*s + '
        '416426942452814092800000)/((15*s - 31)*(12*s - 25)*(11*s - 23)*(9*s - 16)*(9*s - 17)*(9*s - 19)*'
        '(9*s - 20)*(8*s - 15)*(8*s - 17)*(7*s - 13)*(6*s - 11)^3*(5*s - 11)^2*(3*s - 5)*(2*s - 3)*(s - 1)*(s - 2)^5*s)',
        '1/30*(63*s^2 - 130*s + 66)*(15*s - 16)/((9*s - 11)*(7*s - 8)*(5*s - 6)*(3*s - 4)*(s - 1)^3*(s - 2)*s)',
        ),

    ZR( ['C(6,24;0)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 0, 0, 0 ],
                        [ 3, 0, 6, 5, 0, 0 ],
                        [ 5, 6, 0, 0, 0, 0 ],
                        [ 0, 5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/3528*(18318478497368818728960000000*s^30 - 993999468573661828981248000000*s^29 + '
        '26055884746354403433883622400000*s^28 - 439406265493012663314729570240000*s^27 + '
        '5356254303307055494633486019760000*s^26 - 50271114171005244349404277230979200*s^25 + '
        '377849671629715635421127100181210080*s^24 - 2335589486045237356493837292527518368*s^23 + '
        '12098887623935468556734724078767382792*s^22 - 53257097466246665641781192680282841784*s^21 + '
        '201271852073779099370236260650871305586*s^20 - 658164012875528613035973368070783098988*s^19 + '
        '1873011618544262988103905187983038552456*s^18 - 4658176998311757604020009697140474197769*s^17 + '
        '10152832726796496097911658062262809459061*s^16 - 19425013195798828290260921219160486212711*s^15 + '
        '32640990275678875393894850666993552576029*s^14 - 48144662963005083121522820631977846614278*s^13 + '
        '62228383229217492729189391946519561157348*s^12 - 70280920236845748019541625425846464998570*s^11 + '
        '69065337952308025237634721945704678248194*s^10 - 58711814368070086543090117353892835282261*s^9 + '
        '42839002179510474981087968338741060728069*s^8 - 26551451806950980322331870149295377119155*s^7 + '
        '13785944650200736464350968438631034488129*s^6 - 5883885245439268299446636112929410310548*s^5 + '
        '2010067614067392600278138404655778537696*s^4 - 528404438621382698836208667898062701568*s^3 + '
        '100339746135152196976440019485628354560*s^2 - 12249078355994879081419637708692684800*s + '
        '721778584617539142006209265008640000)/((19*s - 39)*(17*s - 32)*(15*s - 31)*(13*s - 25)*(13*s - 28)*'
        '(12*s - 23)*(11*s - 21)*(10*s - 17)*(10*s - 19)^2*(9*s - 16)^2*(9*s - 17)*(9*s - 19)*(8*s - 15)*'
        '(8*s - 17)*(7*s - 11)*(7*s - 13)*(6*s - 11)^2*(5*s - 8)*(5*s - 9)^2*(5*s - 11)*(4*s - 7)^2*(3*s - 5)^2*(2*s - 3)*(s - 1)*(s - 2)^5*s)',
        '1/60*(168*s^2 - 268*s + 105)/((14*s - 15)*(8*s - 7)*(s - 1)^4*(s - 2)*s)',
        ),

    ZR( ['C(6,24;1)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 0, 0 ],
                        [ 3, 0, 6, 5, 0, 0 ],
                        [ 5, 6, 0, 0, 0, 0 ],
                        [ 6, 5, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        None,
        None
        ),

    ZR( ['C(6,25)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 3, 5, 6, 0, 0 ],
                        [ 3, 0, 0, 0, 0, 0 ],
                        [ 5, 0, 0, 0, 0, 0 ],
                        [ 6, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],

            },
        '1/2*(636*s^4 - 4176*s^3 + 10079*s^2 - 10523*s + 3960)/((5*s - 11)*(4*s - 9)*(3*s - 5)*(3*s - 8)*(2*s - 3)*(2*s - 5)*(s - 1)*(s - 2)^2*s)',
        '1/120*(585*s^2 - 1130*s + 528)/((9*s - 11)*(5*s - 6)*(3*s - 4)*(s - 1)^3*(s - 2)*s)',
        ),

    ZR( ['C(6,26)'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 4, 5, 0, 0, 0 ],
                        [ 4, 0, 6, 0, 0, 0 ],
                        [ 5, 6, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ] ],
            },
        '1/540*(262500*s^5 - 2248675*s^4 + 7655715*s^3 - 12954648*s^2 + 10888888*s - 3624192)/((7*s - 16)*(6*s - 13)*(5*s - 11)^2*(s - 1)*(s - 2)^5*s)',
        '2*(12*s^2 - 27*s + 16)/((5*s - 8)*(4*s - 5)*(3*s - 4)^2*(2*s - 3)*(s - 1)*(s - 2)*s)',
        ),

    ####################################
    # OTHER COMMUTATIVE RINGS
    ####################################

    ZR( ['csl(2,ZZ)'],
        {
            'table' : [ [[ 0, 0, 0], [ 0, 0, 1], [ 2, 0, 0]],
                        [[ 0, 0, 1], [ 0, 0, 0], [ 0,-2, 0]],
                        [[ 2, 0, 0], [ 0,-2, 0], [ 0, 0, 0]] ],
            'descr' : 'Commutative version of sl(2,ZZ)',
            },
        '1/3*(294*s^3 - 274*s^2 + 84*s - 9)*(3*s - 1)/((7*s - 3)^2*(2*s - 1)^3*s^2)',
        '0',
        ),

    ZR( ['CLie(2)'],
        {
            'simple_basis': True,
            'matrix': [[0,2],[2,0]],
            'descr': 'Commutative version of Lie(2)',
            },
        '2/((2*s - 1)*s)',
        '1/2/s^2', 
        ),

    ZR( ['Tr(2,ZZ)'],
        {
            'matrix_basis' : [E(2,0,0), E(2,0,1), E(2,1,1)],
            'product' : 'associative',
            'rank': 3,
            'descr' : 'Associative ring of upper triangular 2x2 matrices over ZZ',
            },
        '4/((3*s - 2)*(s - 1)*s)',
        '1/3/s^3'
        ),

    ZR( ['(ZZ^2,*)'],
        {
            'simple_basis' : True,
            'matrix' : [[1,0],[0,2]],
            'descr': 'ZZ^2 with componentwise multiplication'
            },
        '4/((3*s - 1)*s)',
        's^(-2)'
        ),

    ZR( ['ZZ[X]/X^2'],
        {
            'simple_basis' : True,
            'matrix' : [[1,2],
                        [2,0]],
            },
        '2/((2*s - 1)*s)',
        '1/((2*s - 1)*s)'
        ),

    ZR( ['(ZZ^3,*)'],
        {
            'simple_basis' : True,
            'matrix' : [[1,0,0],[0,2,0],[0,0,3]],
            'descr': 'ZZ^3 with componentwise multiplication'
            },
        None,
        's^(-3)',
        ),

    ZR( ['(ZZ^2,*) (+) (ZZ,0)'],
        {
            'simple_basis' : True,
            'matrix' : diagonal_matrix([1,2,0]),
            },
        '2*(20*s^2 - 11*s + 1)/((5*s - 3)*(3*s - 2)*(2*s - 1)*s^2)',
        's^(-3)',
        ),

    ZR( ['(ZZ^2,*) (+) (ZZ^2,0)'],
        {
            'simple_basis': True,
            'matrix' : diagonal_matrix([1,2,0,0]),
            },
        '1/30*(40*s^3 - 76*s^2 + 43*s - 6)/((s - 1)^5*s^2)',
        '1/((s - 1)*s^3)', 
        ),

    ZR( ['ZZ[X]/X^2 (+) (ZZ,0)'],
        {
            'simple_basis' : True,
            'matrix' : [[1,2,0],
                        [2,0,0],
                        [0,0,0]],
            },
        '(4*s - 1)/((4*s - 3)*(s - 1)*s^2)',
        '1/((2*s - 1)*s^2)',
        ),

    ZR( ['ZZ[X]/X^2 (+) (ZZ,*)'],
        {
            'simple_basis' : True,
            'matrix' : [[1,2,0],
                        [2,0,0],
                        [0,0,3]],
            },
        '2*(20*s^2 - 11*s + 1)/((5*s - 3)*(3*s - 2)*(2*s - 1)*s^2)',
        '1/((2*s - 1)*s^2)'
        ),
    
    ZR( ['ZZ[X]/X^3'],
        {
            'simple_basis' : True,
            'matrix' : [[1,2,3],
                        [2,3,0],
                        [3,0,0]],
            },
        '2*(15*s - 8)/((5*s - 4)*(3*s - 2)^2*s)',
        '1/((3*s - 2)*(2*s - 1)*s)'
        ),

    ZR( ['ZZ[X]/X^2 (+) (ZZ^2,0)'],
        {
            'simple_basis' : True,
            'matrix' : [[1,2,0,0],
                        [2,0,0,0],
                        [0,0,0,0],
                        [0,0,0,0]],
            },
        '2*(12*s^2 - 19*s + 6)/((4*s - 5)*(3*s - 4)*(2*s - 3)*(s - 1)*s^2)',
        '1/((2*s - 1)*(s - 1)*s^2)', 
        ),
        
    ZR( ['ZZ[X]/X^3 (+) (ZZ,0)'],
        {
            'simple_basis' : True,
            'matrix' : [[1,2,3,0],
                        [2,3,0,0],
                        [3,0,0,0],
                        [0,0,0,0]],
            },
        '8/5*(26775*s^5 - 108930*s^4 + 173267*s^3 - 134415*s^2 + 50862*s - 7560)/((17*s - 18)*(7*s - 8)*(6*s - 5)*(6*s - 7)*(5*s - 6)*(3*s - 2)*(s - 1)*s^2)',
        '1/((3*s - 2)*(2*s - 1)*s^2)',
        ),

    ZR( ['ZZ[X]/X^3 (+) (ZZ,*)'],
        {
            'simple_basis' : True,
            'matrix' : [[1,2,3,0],
                        [2,3,0,0],
                        [3,0,0,0],
                        [0,0,0,4]],
            },
        None,
        '1/((3*s - 2)*(2*s - 1)*s^2)',
        ),

    ZR( ['ZZ[X]/X^4'],
        {
            'simple_basis' : True,
            'matrix' : [[1,2,3,4],
                        [2,3,4,0],
                        [3,4,0,0],
                        [4,0,0,0]],
            },
        '1/168480*(2021760*s^5 - 8509620*s^4 + 14322332*s^3 - 12036071*s^2 + 5044460*s - 842400)/((6*s - 5)*(4*s - 3)*(s - 1)^6*s)',
        '1/((4*s - 3)*(3*s - 2)*(2*s - 1)*s)'
        ),

    ZR( ['ZZ[X]/X^2 (+) ZZ[X]/X^2'],
        {
            'simple_basis' : True,
            'matrix' : [[1,2,0,0],
                        [2,0,0,0],
                        [0,0,3,4],
                        [0,0,4,0]],
            },
        None, # singular
        '1/((2*s - 1)^2*s^2)',
        ),


    ZR( ['ZZ[X]/X^2 (+) (ZZ,*) (+) (ZZ,0)'],
        {
            'simple_basis' : True,
            'matrix' : [[1,2,0,0],
                        [2,0,0,0],
                        [0,0,3,0],
                        [0,0,0,0]],
            },
        '1/3*(14080*s^6 - 60144*s^5 + 106346*s^4 - 99849*s^3 + 52701*s^2 - 14913*s + 1782)/((11*s - 10)*(10*s - 9)*(4*s - 3)^2*(2*s - 1)*(s - 1)^3*s^2)',
        '1/((2*s - 1)*s^3)',
        ),

    ZR( ['X*ZZ[X]/X^3'],
        {
            'simple_basis' : True,
            'matrix' : [[2,0],
                        [0,0]],
        },
        '2/((3*s - 2)*s)',
        '1/((2*s - 1)*s)',
        ),

    ZR( ['X*ZZ[X]/X^6'],
        {
            'simple_basis' : True,
            'matrix' : [[2,3,4,5,0],
                        [3,4,5,0,0],
                        [4,5,0,0,0],
                        [5,0,0,0,0],
                        [0,0,0,0,0]],
        },
        None,
        None,
        ),
        
    ####################################
    # WEIRD RINGS
    ####################################
    ZR( ['uv'],
        {
            'table' : [[ [ 1, 0, 0], [0, 1, 0], [0, 0, 1]],
                       [ [ 0, 1, 0], [0, 0, 0], [0, 0, 1]],
                       [ [ 0, 0, 1], [0, 0, 1], [0, 0, 0]]],
            },
        '2/3*(45*s - 19)/((5*s - 3)*(3*s - 2)*(2*s - 1)*s)',
        '1/3/((2*s - 1)*s^2)',
        ),

    ZR( ['Cycle(123)'],
        {
            'simple_basis': True,
            'matrix': diagonal_matrix([2,3,1]),
            },
        '1/3*(15*s - 7)*(5*s - 1)/((7*s - 3)*(2*s - 1)^2*s^2)',
        '0'
        ),
    
    ZR( [],
        {
            'rank': 3,
            'table': { (0,0):(1,1,0), (1,1):(0,1,1) },
            },
        '4/3*(30*s^2 - 20*s + 3)/((5*s - 3)*(3*s - 2)*(2*s - 1)*s^2)',
        '1/2/s^3'
        ),

    ZR( [],
        {
            'rank': 4,
            'table': { (0,0):(1,1,0,0), (1,1):(0,1,1,0) },
            },
        '1/45*(60*s^3 - 128*s^2 + 87*s - 18)/((s - 1)^5*s^2)',
        '1/2/((s - 1)*s^3)'
        ),

    ZR( [],
        {
            'rank': 3,
            'table': { (0,1):(1,1,0), (1,0):(1,1,0), (2,2):(0,0,1) },
            },
        '2/3*(125*s^3 - 111*s^2 + 27*s - 1)/((7*s - 3)*(2*s - 1)^3*s^2)',
        '1/2/s^3'
        ),

    ZR( [],
        {
            'rank': 3,
            'table': { (0,1):(1,1,0), (1,0):(1,-1,0), (2,2):(0,0,1) }
            },
        '1/6*(3840*s^4 - 6132*s^3 + 3651*s^2 - 960*s + 94)/((8*s - 3)*(5*s - 2)*(2*s - 1)^4*s)',
        '0'
        ),

    ZR( [],
        {
            'rank': 4,
            'table': { (0,1):(1,1,0,0), (1,0):(1,-1,0,0), (2,2):(0,0,1,0) }
            },
        '1/8*(10897286400*s^10 - 65761563420*s^9 + 177725729355*s^8 -'
        '283096315026*s^7 + 294111735544*s^6 - 208041908802*s^5 +'
        '101349701397*s^4 - 33523685528*s^3 + 7190380152*s^2 - 900424080*s +'
        '49782600)/((13*s - 9)*(11*s - 7)*(9*s - 5)*(7*s - 5)^2*(5*s - 3)*'
        '(5*s - 4)*(4*s - 3)^2*(3*s - 2)^3*(2*s - 1)*s)',
        '0'
        ),

    ####################################
    # MODULES (we only list 'subalgebras'... these are really just submodules)
    ####################################

    # dim 2
    ZR( ['Mod(U_2(ZZ))', 'Mod(U_2)', 'Mod(Triv(2))'],
        {
            'rank' : 2,
            'operators' : [ E(2,0,1) ],
            },
        '1/((2*s - 1)*s)',
        None
        ),
   
    # dim 3
    ZR( ['Mod(U_2 x U_1)'],
        {
            'rank': 3, 
            'operators': [ E(3,0,1) ],
        },
        '1/2/((s - 1)^2*s)',
        None
        ),

    ZR( ['Mod(U_3(ZZ))', 'Mod(U_3)'],
        {
            'rank' : 3,
            'operators' : [ E(3,0,1), E(3,1,2) ],
            },
        '1/2*(4*s - 1)/((3*s - 1)*(2*s - 1)^2*s)',
        None
        ),

    # dim 4
    ZR( ['Mod(U_2 x U_1 x U_1)', 'Mod(U_2 x U_1^2)'],
        {
            'rank': 4, 
            'operators': [ E(4,0,1) ],
        },
        '1/((2*s - 3)*(s - 1)*(s - 2)*s)',
        None
        ),

    ZR( ['Mod(U_2 x U_2)', 'Mod(U_2^2)'],
        {
            'rank': 4, 
            'operators': [ E(4,0,1), E(4,2,3) ],
        },
        '1/12*(3*s - 2)/((s - 1)^4*s)',
        None
        ),

    ZR( ['Mod(U_3 x U_1)'],
        {
            'rank': 4, 
            'operators': [ E(4,0,1), E(4,1,2) ],
        },
        '(2*s - 1)/((4*s - 3)*(3*s - 2)*(s - 1)^2*s)',
        None
        ),

    ZR( ['Mod(U_4(ZZ))', 'Mod(U_4)'],
        {
            'rank' : 4,
            'operators' : [ E(4,0,1), E(4,1,2), E(4,2,3) ],
            },
        '1/8*(3360*s^5 - 5192*s^4 + 3139*s^3 - 930*s^2 + 136*s - 8)/((7*s - 3)*(5*s - 2)*(4*s - 1)*(3*s - 1)^2*(2*s - 1)^3*s)',
        None
        ),

    # dim 5
    ZR( ['Mod(U_2 x U_1 x U_1 x U_1)', 'Mod(U_2 x U_1^3)'],
        {
            'rank': 5, 
            'operators': [ E(5,0,1) ],
        },
        '1/2/((s - 1)*(s - 2)^2*(s - 3)*s)',
        None
        ),

    ZR( ['Mod(U_2^2 x U_1)'],
        {
            'rank': 5,
            'operators': [ E(5,0,1), E(5,2,3) ],
        },
        '3/((3*s - 4)*(2*s - 3)^2*(s - 2)*s)',
        None
        ),

    ZR( ['Mod(U_3 x U_1 x U_1)', 'Mod(U_3 x U_1^2)'],
        {
            'rank': 5, 
            'operators': [ E(5,0,1), E(5,1,2) ],
        },
        '1/12*(4*s - 3)/((2*s - 3)*(s - 1)^3*(s - 2)*s)',
        None
        ),

    ZR( ['Mod(U_3 x U_2)'],
        {
            'rank': 5,
            'operators': [ E(5,0,1), E(5,1,2), E(5,3,4) ],
        },
        '1/36*(180*s^4 - 489*s^3 + 490*s^2 - 216*s + 36)/((5*s - 4)*(4*s - 3)*(3*s - 2)*(s - 1)^5*s)',
        None
        ),


    ZR( ['Mod(U_4 x U_1)'],
        {
            'rank' : 5,
            'operators' : [ E(5,0,1), E(5,1,2), E(5,2,3) ],
            },
        '1/12*(3360*s^5 - 8584*s^4 + 8671*s^3 - 4338*s^2 + 1080*s - 108)/((7*s - 4)*(5*s - 3)*(4*s - 3)^2*(3*s - 2)*(2*s - 1)^2*(s - 1)^2*s)',
        None
        ),

    ZR( ['Mod(U_5(ZZ))', 'Mod(U_5)'],
        {
            'rank' : 5,
            'operators' : [ E(5,0,1), E(5,1,2), E(5,2,3), E(5,3,4) ],
            'descr' : 'Natural module of U_5(ZZ) acting on ZZ^5',
            },
        '(435891456000*s^14 - 1957609382400*s^13 + 4053337786080*s^12 - 5128457632240*s^11 + '
        '4429884320966*s^10 - 2763848527457*s^9 + 1284781950540*s^8 - 452226036325*s^7 + '
        '121188554644*s^6 - 24624421916*s^5 + 3737984412*s^4 - 411498360*s^3 + 31087152*s^2 - '
        '1443744*s + 31104)/(36*(13*s - 6)*(11*s - 4)*(10*s - 3)*(9*s - 4)*(8*s - 3)*'
        '(7*s - 2)*(7*s - 3)*(5*s - 1)*(5*s - 2)^2*(4*s - 1)^2*(3*s - 1)^2*(2*s - 1)^4*s)',
        None
        ),

    # dim 6

    ZR( ['Mod(U_2 x U_1^4)'],
        {
            'rank': 6, 
            'operators': [ E(6,0,1) ],
        },
        '1/((2*s - 5)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s)',
        None
        ),

    ZR( ['Mod(U_2^2 x U_1^2)'],
        {
            'rank': 6,
            'operators': [ E(6,0,1), E(6,2,3) ],
        },
        '1/4*(3*s - 4)/((3*s - 5)*(s - 1)*(s - 2)^3*(s - 3)*s)',
        None
        ),

    ZR( ['Mod(U_2^3)'],
        {
            'rank': 6,
            'operators': [ E(6,0,1), E(6,2,3), E(6,4,5) ],
        },
        '(60*s^3 - 191*s^2 + 201*s - 72)/((5*s - 8)*(4*s - 5)*(3*s - 4)*(2*s - 3)^3*(s - 1)*(s - 2)*s)',
        None
        ),

    ZR( ['Mod(U_3 x U_1^3)'],
        {
            'rank': 6,
            'operators': [ E(6,0,1), E(6,1,2) ],
        },
        '2/((4*s - 5)*(3*s - 4)*(s - 2)^2*(s - 3)*s)',
        None
        ),

    ZR( ['Mod(U_3 x U_2 x U_1)'],
        {
            'rank': 6,
            'operators': [ E(6,0,1), E(6,1,2), E(6,3,4) ],
        },
        '1/60*(180*s^4 - 690*s^3 + 979*s^2 - 612*s + 144)/((3*s - 4)^2*(2*s - 3)^2*(s - 1)^4*(s - 2)*s)',
        None
        ),

    ZR( ['Mod(U_3 x U_3)', 'Mod(U_3^2)'],
        {
            'rank': 6,
            'operators': [ E(6,0,1), E(6,1,2), E(6,3,4), E(6,4,5) ],
        },
        '1/648*(793800*s^9 - 4912110*s^8 + 13409001*s^7 - 21208645*s^6 + 21445130*s^5 - 14400213*s^4 + 6435207*s^3 - 1849878*s^2 + 311040*s - 23328)/((7*s - 5)*(7*s - 6)*(5*s - 3)*(5*s - 4)*(4*s - 3)*(3*s - 2)^2*(s - 1)^7*s)',
        None
        ),

    ZR( ['Mod(U_4 x U_1^2)'],
        {
            'rank' : 6,
            'operators' : [ E(6,0,1), E(6,1,2), E(6,2,3) ],
            },
        '1/32*(1120*s^5 - 3992*s^4 + 5637*s^3 - 3948*s^2 + 1376*s - 192)/((7*s - 5)*(5*s - 4)*(4*s - 3)*(3*s - 2)*(2*s - 3)*(s - 1)^4*(s - 2)*s)',
        None
        ),

    ZR( ['Mod(U_4 x U_2)'],
        {
            'rank': 6, 
            'operators': [ E(6,0,1), E(6,1,2), E(6,2,3), E(6,4,5) ],
        },
        '1/12*(42336000*s^12 - 332018400*s^11 + 1186079760*s^10 - 2552470644*s^9 + 3686349053*s^8 - 3765420755*s^7 + 2790738712*s^6 - 1513116364*s^5 + 596113296*s^4 - 166561128*s^3 + 31358232*s^2 - 3574368*s + 186624)/((10*s - 9)*(9*s - 8)*(8*s - 5)*(7*s - 4)*(7*s - 6)*(5*s - 3)*(5*s - 4)*(4*s - 3)^2*(3*s - 2)*(2*s - 1)^2*(s - 1)^5*s)',
        None
        ),

    ZR( ['Mod(U_5 x U_1)'],
        {
            'rank': 6, 
            'operators': [ E(6,0,1), E(6,1,2), E(6,2,3), E(6,3,4) ],
        },
        '1/48*(479480601600*s^14 - 3353436979200*s^13 + 10835736034176*s^12 - 21441840172880*s^11 + 29031859826547*s^10 - 28457484231924*s^9 + 20830412128130*s^8 - 11570736088224*s^7 + 4903227228015*s^6 - 1578206609388*s^5 + 380029898988*s^4 - 66427903008*s^3 + 7972717248*s^2 - 588425472*s + 20155392)/((13*s - 9)*(11*s - 5)*(11*s - 6)*(9*s - 4)*(8*s - 3)*(7*s - 3)*(7*s - 4)*(5*s - 2)*(5*s - 3)^2*(4*s - 3)^2*(3*s - 2)^3*(2*s - 1)^2*(s - 1)^2*s)',
        None
        ),

    # other

    ZR( ['Mod(U_3 x U_2 x U_2)', 'Mod(U_3 x U_2^2)'],
        {
            'rank': 7,
            'operators': [ E(7,0,1), E(7,1,2), E(7,3,4), E(7,5,6) ],
        },
        '1/240*(201600*s^9 - 1954560*s^8 + 8373080*s^7 - 20814824*s^6 + 33119589*s^5 - 35018799*s^4 + 24638064*s^3 - 11139024*s^2 + 2940480*s - 345600)/((7*s - 10)*(5*s - 8)*(4*s - 5)*(3*s - 4)^2*(2*s - 3)^4*(s - 1)^5*(s - 2)*s)',
        None,
        ),

    ZR( ['Mod(U_2^4)'],
        {
            'rank': 8,
            'operators': [ E(8,0,1), E(8,2,3), E(8,4,5), E(8,6,7) ],
        },
        '1/80*(1050*s^5 - 7045*s^4 + 18885*s^3 - 25452*s^2 + 17360*s - 4800)/((7*s - 15)*(5*s - 7)*(3*s - 5)*(2*s - 3)*(s - 1)*(s - 2)^6*(s - 3)*s)',
        None
        ),

    ZR( ['Mod(b_3(ZZ))'], # = upper triangular matrices , boring computations!
        {
            'rank' : 3,
            'operators' : [ E(3,0,0), E(3,1,1), E(3,2,2), E(3,0,1), E(3,0,2), E(3,1,2) ],
            },
        '1/6/s^3',
        None
        ),

    ZR( ['Mod(b_4(ZZ))'],
        {
            'rank' : 4,
            'operators' : [ E(4,0,0), E(4,1,1), E(4,2,2), E(4,3,3), E(4,0,1), E(4,1,2), E(4,2,3) ],
            },
        '1/24/s^4',
        None
        ),

    ZR( ['Mod(b_5(ZZ))'],
        {
            'rank' : 5,
            'operators' : [ E(5,0,0), E(5,1,1), E(5,2,2), E(5,3,3), E(5,4,4),
                            E(5,0,1), E(5,1,2), E(5,2,3), E(5,3,4) ],
            },
        '1/120/s^5',
        None
        ),

    ZR( ['Mod(nb(sp_4(ZZ))))'],
        {
            'rank' : 4,
            'operators' : [ E(4,0,1) - E(4,3,2),
                            E(4,1,3),
                            E(4,0,3) + E(4,1,2),
                            E(4,0,2) ],
            'descr' : 'Natural module for the nilradical of the Borel subalgebra of sp(4,ZZ)',
            },
        '1/6*(4*s - 1)/((2*s - 1)^4*s)',
        None
        ),

    ####################################
    # GRADED RINGS
    ####################################

    # The following graded nilpotent Lie algebras are taken from
    # O. Kuzmich, Graded nilpotent lie algebras in low dimensions, Lobachevskii J. Math.,
    # 1999, Volume 3, 147--184

    ZR( ['m3_2', 'gr(L(3,2))'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0 ],
                        [ 0, 0, 1 ],
                        [ 0,-1, 0 ] ],
            'blocks': [ 1, 2 ],
        },
        '1/2*(3*s - 1)/((2*s - 1)*(s - 1)*s^2)',
        '1/3/((s - 1)*s^2)',
        ),

    ZR( ['m4_2', 'gr(L(4,2))'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0 ],
                        [ 0, 0, 0, 1 ],
                        [ 0, 0,-1, 0 ] ],
            'blocks': [ 1, 3 ],
        },
        '1/2*(3*s - 1)/((2*s - 1)*(s - 1)*(s - 2)*s^2)',
        '1/3/((s - 1)*(s - 2)*s^2)',
        ),

    ZR( ['m4_3'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0 ],
                        [ 0, 0, 0, 1 ],
                        [ 0, 0, 0, 2 ],
                        [ 0,-1,-2, 0 ] ],
            'blocks': [ 1, 1, 2 ],
        },
        '1/8*(26*s^2 - 17*s + 3)/((3*s - 1)*(2*s - 1)*(s - 1)*s^3)',
        '1/12/((s - 1)*s^3)',
        ),

    ZR( ['m5_2_1'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 2 ],
                        [ 0, 0,-1,-2, 0 ] ],
            'blocks': [ 2, 3 ],
        },
        '1/6*(7*s - 4)/((2*s - 1)*(s - 1)^2*(s - 2)*s^2)',
        '2/5/((3*s - 1)*(s - 1)*(s - 2)*s^2)',
        ),
    
    ZR( ['m5_2_2', 'gr(L(5,2))'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0,-1, 0 ] ],
            'blocks': [ 1, 4 ],
        },
        '1/2*(3*s - 1)/((2*s - 1)*(s - 1)*(s - 2)*(s - 3)*s^2)',
        '1/3/((s - 1)*(s - 2)*(s - 3)*s^2)',
    ),

    ZR( ['m5_2_3', 'gr(L(5,4))'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 1, 0, 0 ],
                        [ 0,-1, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0,-1, 0 ] ],
            'blocks': [ 1, 4 ],
        },
        '1/9*(17*s - 6)/((3*s - 2)*(s - 1)*(s - 2)*(s - 3)*s^2)',
        '1/5/((s - 1)*(s - 2)*(s - 3)*s^2)',
    ),
    
    ZR( ['m5_3_1'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 1, 2 ],
                        [ 0, 0,-1, 0, 3 ],
                        [ 0, 0,-2,-3, 0 ] ],
            'blocks': [ 2, 1, 2 ],
        },
        '1/5*(121*s^3 - 127*s^2 + 40*s - 4)/((5*s - 1)*(3*s - 2)*(2*s - 1)^2*(s - 1)*s^3)',
        '2/15/((4*s - 1)*(s - 1)*s^3)',
    ),

    ZR( ['m5_3_2'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 2 ],
                        [ 0,-1, 0,-2, 0 ] ],
            'blocks': [ 1, 1, 3 ],
        },
        '1/8*(26*s^2 - 17*s + 3)/((3*s - 1)*(2*s - 1)*(s - 1)*(s - 2)*s^3)',
        '1/12/((s - 1)*(s - 2)*s^3)',
    ),

    ZR( ['m5_4_1'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 2 ],
                        [ 0, 0, 0, 0, 3 ],
                        [ 0,-1,-2,-3, 0 ] ],
            'blocks': [ 1, 1, 1, 2 ],
        },
        '1/84*(828*s^3 - 776*s^2 + 247*s - 26)/((4*s - 1)*(3*s - 1)*(2*s - 1)*(s - 1)*s^4)',
        '1/60/((s - 1)*s^4)',
    ),

    ZR( ['m6_2_1'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 3,-2 ],                        
                        [ 0, 0, 0,-3, 0, 1 ],
                        [ 0, 0, 0, 2,-1, 0 ] ],
            'blocks': [ 3, 3 ],
        },
        '1/3*(25*s^2 - 28*s + 6)/((3*s - 1)*(3*s - 2)*(2*s - 3)*(s - 1)^2*(s - 2)*s^2)',
        '1/((5*s - 2)*(3*s - 2)*(s - 1)*(s - 2)*s^2)',
    ),


    ZR( ['m6_2_2'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 0, 2 ],
                        [ 0, 0, 0,-1,-2, 0 ] ],
            'blocks': [ 2, 4 ],
        },
        '1/6*(7*s - 4)/((2*s - 1)*(s - 1)^2*(s - 2)*(s - 3)*s^2)',
        '2/5/((3*s - 1)*(s - 1)*(s - 2)*(s - 3)*s^2)',
    ),

    ZR( ['m6_2_3'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 1, 0, 0 ],
                        [ 0, 0,-1, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 2 ],
                        [ 0, 0, 0, 0,-2, 0 ] ],
            'blocks': [ 2, 4 ],
        },
        '1/12*(243*s^4 - 685*s^3 + 671*s^2 - 279*s + 42)/((3*s - 1)*(3*s - 4)*(2*s - 1)^2*(s - 1)^2*(s - 2)*(s - 3)*s^2)',
        '5/9/((5*s - 1)*(s - 1)*(s - 2)*(s - 3)*s^2)',
    ),

    ZR( ['m6_2_4'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0,-1, 0 ],
                        [ 0, 0, 0, 1, 0, 2 ],
                        [ 0, 0,-1, 0,-2, 0 ] ],
            'blocks': [ 2, 4 ],
        },
        '1/15*(2144*s^5 - 6886*s^4 + 8164*s^3 - 4499*s^2 + 1155*s - 108)/((4*s - 1)*(4*s - 3)*(3*s - 1)*(3*s - 4)*(2*s - 1)*(s - 1)^2*(s - 2)*(s - 3)*s^2)',
        '1/3*(8*s - 1)/((6*s - 1)*(5*s - 1)*(s - 1)*(s - 2)*(s - 3)*s^2)',
    ),

    ZR( ['m6_2_5'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0,-1, 0 ] ],
            'blocks': [ 1, 5 ],
        },
        '1/2*(3*s - 1)/((2*s - 1)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s^2)',
        '1/3/((s - 1)*(s - 2)*(s - 3)*(s - 4)*s^2)',
    ),

    ZR( ['m6_2_6'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 1, 0, 0 ],
                        [ 0, 0,-1, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0,-1, 0 ] ],
            'blocks': [ 1, 5 ],
        },
        '1/9*(17*s - 6)/((3*s - 2)*(s - 1)*(s - 2)*(s - 3)*(s - 4)*s^2)',
        '1/5/((s - 1)*(s - 2)*(s - 3)*(s - 4)*s^2)',
    ),

    ZR( ['m6_3_1'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 1, 2 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0,-1, 0, 0, 3 ],
                        [ 0, 0,-2, 0,-3, 0 ] ],
            'blocks': [ 2, 1, 3 ],
        },
        '1/5*(121*s^3 - 127*s^2 + 40*s - 4)/((5*s - 1)*(3*s - 2)*(2*s - 1)^2*(s - 1)*(s - 2)*s^3)',
        '2/15/((4*s - 1)*(s - 1)*(s - 2)*s^3)',
    ),

    ZR( ['m6_3_2'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 0, 2 ],
                        [ 0, 0, 0, 0, 0, 3 ],
                        [ 0, 0,-1,-2,-3, 0 ] ],
            'blocks': [ 1, 2, 3 ],
        },
        '1/120*(318*s^2 - 287*s + 59)/((3*s - 1)*(2*s - 1)*(s - 1)^2*(s - 2)*s^3)',
        '13/120/((3*s - 1)*(s - 1)*(s - 2)*s^3)',
    ),

    ZR( ['m6_3_3'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 1, 0 ],
                        [ 0, 0, 0, 0, 0, 2 ],
                        [ 0, 0,-1, 0, 0, 3 ],
                        [ 0, 0, 0,-2,-3, 0 ] ],
            'blocks': [ 1, 2, 3 ],
        },
        '1/96*(243*s^2 - 217*s + 46)/((3*s - 1)*(2*s - 1)*(s - 1)^2*(s - 2)*s^3)',
        '13/120/((3*s - 1)*(s - 1)*(s - 2)*s^3)',
    ),

    ZR( ['m6_3_4'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 1, 0 ],
                        [ 0, 0, 0, 0, 0, 2 ],
                        [ 0, 0,-1, 0, 0, 3 ],
                        [ 0,-1, 0,-2,-3, 0 ] ],
            'blocks': [ 1, 2, 3 ],
        },
        '1/288*(299760*s^5 - 433196*s^4 + 233416*s^3 - 58627*s^2 + 6965*s - 318)/((8*s - 3)*(6*s - 1)^2*(5*s - 1)*(2*s - 1)*(s - 1)^2*(s - 2)*s^3)',
        '1/15/((3*s - 1)*(s - 1)*(s - 2)*s^3)',
    ),

    ZR( ['m6_3_5'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 1, 0 ],
                        [ 0, 0, 0, 1, 0, 0 ],
                        [ 0, 0,-1, 0, 0, 2 ],
                        [ 0,-1, 0, 0, 0, 3 ],
                        [ 0, 0, 0,-2,-3, 0 ] ],
            'blocks': [ 1, 2, 3 ],
        },
        '1/36*(1280*s^3 - 1396*s^2 + 457*s - 41)/((8*s - 3)*(6*s - 1)*(2*s - 1)*(s - 1)^2*(s - 2)*s^3)',
        '1/15/((3*s - 1)*(s - 1)*(s - 2)*s^3)',
    ),

    ZR( ['m6_3_6'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 2 ],
                        [ 0,-1, 0, 0,-2, 0 ] ],
            'blocks': [ 1, 1, 4 ],
        },
        '1/8*(26*s^2 - 17*s + 3)/((3*s - 1)*(2*s - 1)*(s - 1)*(s - 2)*(s - 3)*s^3)',
        '1/12/((s - 1)*(s - 2)*(s - 3)*s^3)',
    ),

    ZR( ['m6_4_1'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 2, 3 ],
                        [ 0, 0, 0,-2, 0, 4 ],
                        [ 0, 0,-1,-3,-4, 0 ] ],
            'blocks': [ 1, 2, 1, 2 ],
        },
        '1/3360*(1481688*s^5 - 2219166*s^4 + 1252147*s^3 - 338066*s^2 + 44015*s - 2218)/((6*s - 1)*(5*s - 1)*(4*s - 1)*(3*s - 2)*(2*s - 1)^2*(s - 1)*s^4)',
        '1/30/((4*s - 1)*(s - 1)*s^4)',
    ),

    ZR( ['m6_4_2'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 1, 0 ],
                        [ 0, 0, 0, 0, 2, 3 ],
                        [ 0, 0,-1,-2, 0, 4 ],
                        [ 0,-1, 0,-3,-4, 0 ] ],
            'blocks': [ 1, 2, 1, 2 ],
        },
        '1/2205*(874464*s^5 - 1278271*s^4 + 696618*s^3 - 180619*s^2 + 22698*s - 1120)/((6*s - 1)*(5*s - 1)*(4*s - 1)*(3*s - 2)*(2*s - 1)^2*(s - 1)*s^4)',
        '1/45/((4*s - 1)*(s - 1)*s^4)',
    ),

    ZR( ['m6_4_3'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 0, 2 ],
                        [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 3 ],
                        [ 0,-1,-2, 0,-3, 0 ] ],
            'blocks': [ 1, 1, 1, 3 ],
        },
        '1/84*(828*s^3 - 776*s^2 + 247*s - 26)/((4*s - 1)*(3*s - 1)*(2*s - 1)*(s - 1)*(s - 2)*s^4)',
        '1/60/((s - 1)*(s - 2)*s^4)',
    ),

    ZR( ['m6_5_1'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 0, 1 ],
                        [ 0, 0, 0, 0, 0, 2 ],
                        [ 0, 0, 0, 0, 0, 3 ],
                        [ 0, 0, 0, 0, 0, 4 ],
                        [ 0,-1,-2,-3,-4, 0 ] ],
            'blocks': [ 1, 1, 1, 1, 2 ],
        },
        '1/9240*(366480*s^4 - 429078*s^3 + 186575*s^2 - 35352*s + 2455)/((5*s - 1)*(4*s - 1)*(3*s - 1)*(2*s - 1)*(s - 1)*s^5)',
        '1/360/((s - 1)*s^5)',
    ),

    ZR( ['m6_5_2'],
        {
            'simple_basis': True,
            'matrix': [ [ 0, 0, 0, 0, 0, 0 ],
                        [ 0, 0, 0, 0, 1, 0 ],
                        [ 0, 0, 0,-1, 0, 2 ],
                        [ 0, 0, 1, 0, 0, 3 ],
                        [ 0,-1, 0, 0, 0, 4 ],
                        [ 0, 0,-2,-3,-4, 0 ] ],
            'blocks': [ 1, 1, 1, 1, 2 ],
        },
        '1/151200*(4512840*s^4 - 5153558*s^3 + 2205667*s^2 - 413503*s + 28502)/((5*s - 1)*(4*s - 1)*(3*s - 1)*(2*s - 1)*(s - 1)*s^5)',
        '1/360/((s - 1)*s^5)',
    ),
    ]

# Take algebras from raw and tensor them with dual numbers.
_raw_tensored_with_duals = [
    ZR(['L(4,3)[eps]'], 'L(4,3)', None, None),

    ZR(['L(5,4)[eps]'], 'L(5,4)', None, None),
    ZR(['L(5,5)[eps]'], 'L(5,5)', None, None),
    ZR(['L(5,6)[eps]'], 'L(5,6)', None, None),
    ZR(['L(5,7)[eps]'], 'L(5,7)', None, None),
    ZR(['L(5,8)[eps]'], 'L(5,8)', None, None),
    ZR(['L(5,9)[eps]'], 'L(5,9)', None, None),

    ZR(['L(6,10)[eps]'], 'L(6,10)', None, None),
    ZR(['L(6,11)[eps]'], 'L(6,11)', None, None),
    ZR(['L(6,12)[eps]'], 'L(6,12)', None, None),
    ZR(['L(6,13)[eps]'], 'L(6,13)', None, None),
    ZR(['L(6,14)[eps]'], 'L(6,14)', None, None),
    ZR(['L(6,15)[eps]'], 'L(6,15)', None, None),
    ZR(['L(6,16)[eps]'], 'L(6,16)', None, None),
    ZR(['L(6,17)[eps]'], 'L(6,17)', None, None),
    ZR(['L(6,18)[eps]'], 'L(6,18)', None, None),
    ZR(['L(6,19;0)[eps]'], 'L(6,19;0)', None, None),
    ZR(['L(6,19;1)[eps]'], 'L(6,19;1)', None, None),
    ZR(['L(6,20)[eps]'], 'L(6,20)', None, None),
    ZR(['L(6,21;0)[eps]'], 'L(6,21;0)', None, None),
    ZR(['L(6,21;1)[eps]'], 'L(6,21;1)', None, None),
    ZR(['L(6,22;0)[eps]'], 'L(6,22;0)', None, None),
    ZR(['L(6,22;1)[eps]'], 'L(6,22;1)', None, None),
    ZR(['L(6,23)[eps]'], 'L(6,23)', None, None),
    ZR(['L(6,24;0)[eps]'], 'L(6,24;0)', None, None),
    ZR(['L(6,24;1)[eps]'], 'L(6,24;1)', None, None),
    ZR(['L(6,24;0)[eps]'], 'L(6,24;0)', None, None),
    ZR(['L(6,25)[eps]'], 'L(6,25)', None, None),
    ZR(['L(6,26)[eps]'], 'L(6,26)', None, None),

    ZR(['g(37A)[eps]'], 'g(37A)', None, None),
    ZR(['g(37C)[eps]'], 'g(37C)', None, None),

    ZR(['g(27A)[eps]'], 'g(27A)', None, None),
    ZR(['g(27B)[eps]'], 'g(27B)', None, None),

    ZR(['g(17)[eps]'], 'g(17)', None, None),
    ZR(['g(157)[eps]'], 'g(157)', None, None),

    ZR(['N_3^(8,2)[eps]'], 'N_3^(8,2)', None, None),
    ZR(['N_4^(8,2)[eps]'], 'N_4^(8,2)', None, None),
    ]

# Topological representation zeta functions associated with unipotent
# groups/nilpotent Lie algebras.
_reps = {
    ####################################
    # de Graaf's tables
    ####################################

    'L(1,1)': '1',

    'L(3,2)': 's/(s - 1)',

    'L(4,3)': 's^2/(s - 1)^2',

    'L(5,4)': '2*s/(2*s - 1)',
    'L(5,5)': '1/2*(2*s - 1)*s/(s - 1)^2',
    'L(5,6)': '2*s^2/((2*s - 3)*(s - 1))',
    'L(5,7)': 's^2/((s - 1)*(s - 2))',
    'L(5,8)': 's/(s - 2)',
    'L(5,9)': 's^2/((s - 1)*(s - 2))',

    'L(6,10)': '2*s^2/((2*s - 1)*(s - 1))',
    'L(6,11)': '1/2*(6*s + 1)*s/((3*s - 4)*(s - 1))',
    'L(6,12)': '2*s^2/((2*s - 3)*(s - 1))',
    'L(6,13)': '1/6*(12*s^2 - 18*s + 7)*s^2/((2*s - 1)*(s - 1)^3)',
    'L(6,14)': '1/3*(12*s^2 - 12*s + 1)*s^2/((2*s - 1)*(2*s - 3)*(s - 1)^2)',
    'L(6,15)': '(6*s - 7)*s^2/((3*s - 5)*(2*s - 3)*(s - 1))',
    'L(6,16)': '2*s^3/((2*s - 1)*(s - 1)*(s - 2))',
    'L(6,17)': '1/2*(2*s - 3)*s^2/((s - 1)*(s - 2)^2)',
    'L(6,18)': 's^2/((s - 1)*(s - 3))',
    'L(6,19;0)': 's^2/((s - 1)*(s - 2))',
    'L(6,19;1)': '2*s^2/((2*s - 1)*(s - 2))',
    'L(6,20)': '1/2*(2*s - 1)*s/((s - 1)*(s - 2))',
    'L(6,21;0)': 's^2/(s - 2)^2',
    'L(6,21;1)': '2*s^2/((2*s - 3)*(s - 2))',
    'L(6,22;0)': '2*s/(2*s - 3)',
    'L(6,22;1)': 's^2/(s - 1)^2',
    'L(6,23)': '1/2*(2*s - 3)*s/(s - 2)^2',
    'L(6,24;0)': '(4*s^2 - 6*s + 1)*s/((2*s - 3)^2*(s - 1))',
    'L(6,24;1)': '(2*s + 1)*s/((2*s - 3)*(s - 1))',
    'L(6,25)': '(s - 1)*s/(s - 2)^2',
    'L(6,26)': 's/(s - 3)',

    ####################################
    # Seeley's tables
    ####################################

    ####################################
    'g(37A)': 's/(s - 3)',
    'g(37B)': '(s - 1)*s/(s - 2)^2',
    'g(37C)': '2*s/(2*s - 5)',
    'g(37D)': '2*(s - 1)*s/((2*s - 3)*(s - 2))',
    ####################################

    ####################################
    'g(357A)': '(s - 1)*s/((s - 2)*(s - 3))',
    'g(357B)': '(s - 1)^2*s/(s - 2)^3',
    'g(357C)': '1/2*(2*s - 3)*(s - 1)*s/(s - 2)^3',
    ####################################

    ####################################
    'g(27A)': '1/2*(2*s - 1)*s/(s - 1)^2',
    'g(27B)': 's/(s - 1)',
    ####################################

    ####################################
    'g(257A)': '2*(s - 1)*s/((2*s - 3)*(s - 2))',
    'g(257B)': '(4*s^2 - 6*s + 1)*s/((2*s - 3)^2*(s - 1))',
    'g(257C)': '2*(s - 1)*s/((2*s - 3)*(s - 2))',
    'g(257D)': '4*(s - 1)*s/(2*s - 3)^2',
    'g(257E)': '1/2*(2*s - 1)*s/((s - 1)*(s - 2))',
    'g(257F)': '2*(s - 1)*s/((2*s - 3)*(s - 2))',
    'g(257G)': '1/2*(12*s^2 - 22*s + 9)*s/((3*s - 4)*(2*s - 3)*(s - 1))', # wt 1
    'g(257H)': '1/4*(4*s^2 - 2*s + 1)*s/(s - 1)^3',
    'g(257I)': '2*s^2/((2*s - 3)*(s - 1))',
    'g(257J)': '(2*s - 1)*s/((2*s - 3)*(s - 1))',
    'g(257K)': '1/2*(2*s - 1)*s^2/(s - 1)^3',
    'g(257L)': '1/4*(2*s - 1)^2*s/(s - 1)^3',
    ####################################

    ####################################
    'g(247A)': '(s - 1)*s/((s - 2)*(s - 3))',
    'g(247B)': 's^2/(s - 2)^2',
    'g(247C)': '1/2*(2*s - 3)*(s - 1)*s/(s - 2)^3',
    'g(247D)': '1/2*(2*s - 1)*s/(s - 2)^2',
    'g(247E)': '1/2*(2*s - 1)*s/(s - 2)^2',
    'g(247F)': 's^3/((s - 1)^2*(s - 2))',
    'g(247G)': '1/4*(4*s^2 - 2*s + 1)*s/((s - 1)^2*(s - 2))',
    'g(247H)': '1/2*(2*s^2 - 2*s + 1)*s/((s - 1)^2*(s - 2))',
    'g(247I)': '2*s^2/((2*s - 3)*(s - 2))', # wt 1
    'g(247J)': '1/2*(6*s + 1)*s/((3*s - 4)*(s - 2))',  # wt 1
    'g(247K)': '(4*s - 1)*s/((4*s - 5)*(s - 2))', # wt 1
    'g(247L)': '2*(s - 1)*s/((2*s - 5)*(s - 2))',
    'g(247M)': '(2*s^2 - 4*s + 1)*s/((2*s - 3)*(s - 2)^2)',
    'g(247N)': '2*(s - 1)*s/((2*s - 3)*(s - 3))',
    'g(247O)': '1/6*(12*s^2 - 42*s + 37)*(s - 1)*s/((2*s - 3)*(s - 2)^3)', # wt 1
    'g(247P)': '2*(s - 1)*s/((2*s - 3)*(s - 3))',
    'g(247Q)': '1/2*(6*s^2 - 14*s + 7)*s/((3*s - 4)*(s - 2)^2)', # wt 1
    'g(247R)': '1/2*(6*s^2 - 14*s + 7)*s/((3*s - 4)*(s - 2)^2)', # wt 2
    ####################################

    ####################################
    'g(2457A)': '(s - 1)*s/((s - 2)*(s - 3))',
    'g(2457B)': 's^2/(s - 2)^2',
    'g(2457C)': '2*(s - 1)*s/((2*s - 5)*(s - 2))',
    'g(2457D)': '1/2*(2*s - 3)*(s - 1)*s/(s - 2)^3', # wt 1
    'g(2457E)': '(2*s + 1)*s/((2*s - 3)*(s - 2))',
    'g(2457F)': '1/2*(2*s - 3)*(s - 1)*s/(s - 2)^3',
    'g(2457G)': '(2*s^2 - 4*s + 1)*s/((2*s - 3)*(s - 2)^2)',
    'g(2457H)': '(4*s^2 - 6*s + 1)*s/((2*s - 3)^2*(s - 2))',
    'g(2457I)': '(4*s^2 - 6*s + 1)*s/((2*s - 3)^2*(s - 2))',
    'g(2457J)': '2*(6*s^2 - 7*s - 2)*s/((3*s - 5)*(2*s - 3)^2)', # wt 1
    'g(2457K)': '2*(24*s^3 - 82*s^2 + 84*s - 23)*s/((4*s - 7)*(3*s - 5)*(2*s - 3)^2)', # wt 1
    'g(2457L)': '(s - 1)*s^2/(s - 2)^3',
    'g(2457M)': '2*s^2/((2*s - 5)*(s - 2))', # wt 1
    ####################################

    ####################################
    'g(2357A)': '2*s^2/((2*s - 3)*(s - 2))', # wt 3
    'g(2357B)': '1/6*(12*s^2 - 42*s + 37)*(s - 1)*s/((2*s - 3)*(s - 2)^3)', # wt 1
    'g(2357C)': '1/2*(72*s^3 - 192*s^2 + 134*s - 9)*s/((3*s - 4)^2*(2*s - 3)^2)', # wt 1
    'g(2357D)': '1/2*(36*s^2 - 24*s - 29)*s/((3*s - 4)^2*(2*s - 3))', # wt 2
    ####################################

    ####################################
    'g(23457A)': 's^2/((s - 2)*(s - 3))',
    'g(23457B)': '2*(s - 1)*s^2/((2*s - 3)*(s - 2)^2)', # wt 1
    'g(23457C)': '1/2*(2*s - 1)*s^2/((s - 1)^2*(s - 3))',
    'g(23457D)': '1/2*(6*s^3 - 22*s^2 + 25*s - 8)*s^2/((3*s - 5)*(s - 1)^2*(s - 2)^2)', # wt 2
    'g(23457E)': '2*s^2/((2*s - 5)*(s - 2))', # wt 3
    # 'g(23457F)': None, # spread collision
    # 'g(23457G)': None, # spread collision
    ####################################

    ####################################
    'g(17)': '3*s/(3*s - 1)',
    ####################################

    ####################################
    'g(157)': '(3*s - 1)*s/((3*s - 2)*(s - 1))',
    ####################################

    ####################################
    'g(147A)': '1/3*(3*s - 2)*s/((s - 1)*(s - 2))',
    'g(147B)': '1/3*(3*s - 2)*s/((s - 1)*(s - 2))',
    'g(147C)': '3*(s - 1)*s/((3*s - 4)*(s - 3))',
    'g(147D)': '3*(s - 1)*s/((3*s - 4)*(s - 3))',
    #'g(147E:0)': '2*(s - 1)*s/((2*s - 3)*(s - 3))',
    #'g(147E:1)': '2*(s - 1)*s/((2*s - 3)*(s - 3))',
    ####################################

    ####################################
    'g(1457A)': '1/2*(2*s - 1)*s^2/(s - 1)^3',
    'g(1457B)': '1/3*(3*s - 2)*s^2/(s - 1)^3',
    ####################################

    ####################################
    'g(137A)': '2*s^3/((2*s - 1)*(s - 1)^2)',
    'g(137B)': '(9*s^3 - 9*s^2 + 3*s - 1)*s/((3*s - 2)^2*(s - 1)^2)',
    'g(137C)': '4*s^2/((2*s - 1)*(2*s - 3))',
    'g(137D)': '6*(2*s - 1)*s/((6*s - 5)*(2*s - 3))',
    ####################################

    ####################################
    'g(1357A)': '1/6*(3*s - 2)*(2*s - 1)*s/(s - 1)^3', # wt 1
    'g(1357B)': '1/8*(16*s^3 - 24*s^2 + 8*s + 1)*s/((2*s - 1)*(s - 1)^3)',
    'g(1357C)': '1/8*(48*s^3 - 96*s^2 + 60*s - 11)*s/((6*s - 5)*(s - 1)^3)',
    'g(1357D)': '3*(s - 1)^2*s/((3*s - 4)*(s - 2)^2)',
    'g(1357E)': '2*(s - 1)^2*s/((2*s - 3)*(s - 2)^2)',
    'g(1357F)': '3*(s - 1)^2*s/((3*s - 4)*(s - 2)^2)',
    'g(1357G)': '2*s^2/((2*s - 3)*(s - 2))',
    'g(1357H)': '3*s^2/((3*s - 4)*(s - 2))',  # wt 1
    'g(1357I)': '1/2*(2*s - 1)*s^2/((s - 1)^2*(s - 2))',
    'g(1357J)': '1/9*(3*s - 1)^2*s/((s - 1)^2*(s - 2))',
    'g(1357K)': '3/2*(2*s - 3)*(s - 1)*s/((3*s - 4)*(s - 2)^2)', # wt 1
    'g(1357L)': '3/2*(2*s - 3)*(s - 1)*s/((3*s - 4)*(s - 2)^2)', # wt 1
    'g(1357M:0)': '1/6*(12*s^2 - 42*s + 37)*(s - 1)*s/((2*s - 3)*(s - 2)^3)', # wt 1
    'g(1357M:1)': '(s - 1)*s/(s - 2)^2',
    'g(1357N:0)': '(s - 1)*s/(s - 2)^2',
    'g(1357N:1)': '3/2*(2*s - 3)*(s - 1)*s/((3*s - 4)*(s - 2)^2)', # wt 1
    'g(1357O)': '2*(4*s^2 - 6*s + 1)*s/(2*s - 3)^3',
    'g(1357P)': '3*(4*s^2 - 6*s + 1)*s/((3*s - 4)*(2*s - 3)^2)', # wt 1
    'g(1357Q)': '2*(2*s + 1)*s/(2*s - 3)^2',
    'g(1357R)': '3*(2*s + 1)*s/((3*s - 4)*(2*s - 3))', # wt 1
    'g(1357S:0)': '3*(2*s + 1)*s/((3*s - 4)*(2*s - 3))', # wt 1
    #'g(1357S:1)': '1/2*(36*s^2 - 24*s - 29)*s/((3*s - 4)^2*(2*s - 3))', # wt 2
    'g(1357S:-1)': '3*(2*s + 1)*s/((3*s - 4)*(2*s - 3))', # wt 1
    ####################################

    ####################################
    'g(13457A)': '1/2*(2*s - 3)*s^2/((s - 1)*(s - 2)^2)',
    'g(13457B)': '(6*s^3 - 16*s^2 + 10*s + 1)*s/((3*s - 5)*(2*s - 3)*(s - 1)*(s - 2))',
    'g(13457C)': '(s - 1)*s/((s - 2)*(s - 3))',
    'g(13457D)': '1/2*(16*s^3 - 40*s^2 + 24*s + 1)*s/((2*s - 3)^3*(s - 1))', # wt 1
    'g(13457E)': '1/3*(18*s^3 - 24*s^2 + 10*s - 3)*s/((3*s - 2)*(2*s - 3)*(s - 1)^2)', # wt 1
    'g(13457F)': '1/2*(2*s - 3)*s^2/(s - 2)^3',
    'g(13457G)': '(3*s - 4)*s^2/((3*s - 5)*(s - 2)^2)', # wt 1
    # 'g(13457H)': None # not a Lie algebra!
    'g(13457I)': '(3*s - 4)*s^2/((3*s - 5)*(s - 2)^2)', # wt 1
    ####################################

    ####################################
    'g(12457A)': '4*s^2/(2*s - 3)^2',
    'g(12457B)': '4*s^2/(2*s - 3)^2', # weight 3
    'g(12457C)': '4*s^3/((2*s - 1)*(2*s - 3)*(s - 1))',
    'g(12457D)': '2*(18*s^3 - 21*s^2 + 8*s - 3)*s/((6*s - 5)*(3*s - 2)*(2*s - 3)*(s - 1))',
    'g(12457E)': '(288*s^3 - 624*s^2 + 266*s + 81)*s/((6*s - 7)*(4*s - 5)^2*(3*s - 4))', # wt 2
    'g(12457F)': '1/8*(48*s^3 - 40*s^2 - 8*s + 3)*s/((3*s - 4)*(2*s - 1)*(s - 1)^2)',
    'g(12457G)': '1/8*(432*s^4 - 864*s^3 + 540*s^2 - 139*s + 34)*s/((6*s - 5)*(3*s - 2)*(3*s - 4)*(s - 1)^2)',
    'g(12457H)': '4*(s - 1)*s^2/((2*s - 3)^2*(s - 2))',
    'g(12457I)': '2/3*(6*s - 7)*s^2/((2*s - 3)^2*(s - 2))', # wt 2
    'g(12457J)': '2/3*(6*s - 7)*s^2/((2*s - 3)^2*(s - 2))', # wt 2
    'g(12457K)': '4*(s - 1)*s^2/((2*s - 3)^2*(s - 2))',
    # 'g(12457M)': None, # spread collision
    # 'g(12457N:1)': None, # depth
    # 'g(12457N:-1)': None, # spread collision
    ####################################

    ####################################
    'g(12357A)': '1/8*(8*s^2 - 12*s + 5)*s^2/(s - 1)^4', # wt 3
    'g(12357B)': '1/8*(8*s^2 - 12*s + 5)*s^2/(s - 1)^4', # wt 3
    'g(12357C)': '1/56*(3024*s^5 - 10248*s^4 + 13286*s^3 - 7893*s^2 + 1900*s - 64)*s/((9*s - 8)*(6*s - 5)*(s - 1)^4)',  # wt 3
    ####################################

    ####################################
    'g(123457A)': 's^2/((s - 1)*(s - 4))',
    'g(123457B)': '2*(s - 2)*s^2/((2*s - 5)*(s - 1)*(s - 3))',
    'g(123457C)': '(3*s - 4)*s^2/((3*s - 5)*(s - 1)*(s - 3))',
    'g(123457D)': '(2*s - 3)*s^2/((2*s - 5)*(s - 1)*(s - 2))', # wt 1
    'g(123457E)': '(2*s - 3)*s^2/((2*s - 5)*(s - 1)*(s - 2))', # wt 2
    'g(123457F)': '1/2*(3*s - 4)*(2*s - 3)*s^2/((3*s - 5)*(s - 1)*(s - 2)^2)', # wt 3 
    'g(123457G)': '(4*s - 5)*s^2/((4*s - 7)*(s - 1)*(s - 2))', # wt 2
    'g(123457H)': '(4*s - 5)*s^2/((4*s - 7)*(s - 1)*(s - 2))',
    # 'g(123457I:0)': None # spread collision
    ####################################

    ####################################
    'L(4,3)[eps]': '2*(4*s^2 - 6*s + 1)*s/(2*s - 3)^3', # wt 1
    'L(5,4)[eps]': '4*s/(4*s - 3)',
    'L(5,5)[eps]': '16*(s - 1)^2*s/((4*s - 5)*(2*s - 3)^2)', # wt 2
    'L(5,6)[eps]': None, # depth
    'L(5,7)[eps]': '(2*s^2 - 4*s + 1)*s/((2*s - 5)*(s - 2)^2)', # wt 3
    'L(5,8)[eps]': '(2*s - 3)*s/((2*s - 5)*(s - 2))', # wt 1
    'L(5,9)[eps]': '(2*s^2 - 4*s + 1)*s/((2*s - 5)*(s - 2)^2)',
    'L(6,10)[eps]': '2/3*(12*s^2 - 12*s + 1)*s/((4*s - 3)*(2*s - 3)*(s - 1))', # wt 1
    'L(6,22;0)[eps]': '2*(2*s - 3)*s/((4*s - 7)*(s - 2))', # wt 2
    'L(6,23)[eps]': '2*(8*s^3 - 44*s^2 + 82*s - 53)*s/((4*s - 9)*(2*s - 5)^2*(s - 2))', # wt 5
    'L(6,24;0)[eps]': '2*(512*s^6 - 4544*s^5 + 16544*s^4 - 31500*s^3 + 32885*s^2 - 17685*s + 3769)*s/((8*s - 15)*(4*s - 7)^3*(2*s - 3)*(s - 2)^2)',
    'L(6,25)[eps]': '2*(4*s^3 - 20*s^2 + 33*s - 19)*s/((2*s - 5)^3*(s - 2))', # wt 3
    'L(6,26)[eps]': '2*(s - 2)*s/((2*s - 7)*(s - 3))', # wt 3
    'g(37A)[eps]': '2*(s - 2)*s/((2*s - 7)*(s - 3))', # wt 3
    'g(37C)[eps]': '4*(s - 2)*s/((4*s - 11)*(s - 3))', # wt 5
    'g(27A)[eps]': '2*(12*s^2 - 22*s + 9)*s/((4*s - 5)*(3*s - 4)*(2*s - 3))', # wt 1
    'g(27B)[eps]': '(20*s - 19)*s/((5*s - 6)*(4*s - 5))',
    'g(17)[eps]': '2*s/(2*s - 1)',
    'g(157)[eps]': '4/3*(3*s - 2)^2*s/((6*s - 5)*(2*s - 3)*(s - 1))', # wt 2
    ####################################

    ####################################
    'N_1^(8,2)': '6*s^2/((3*s - 2)*(2*s - 1))', # wt 1
    'N_2^(8,2)': 's/(s - 1)',
    'N_3^(8,2)': '2*(4*s - 1)*s/((4*s - 3)*(2*s - 1))',
    'N_4^(8,2)': '1/3*(3*s - 2)*s/(s - 1)^2',
    'N_5^(8,2)': '6*s/(6*s - 5)',

    'N_1^(8,3)': '2*(s - 1)*s/((2*s - 3)*(s - 2))',
    'N_2^(8,3)': '2*(s - 1)*s/((2*s - 3)*(s - 2))',
    'N_3^(8,3)': '(4*s^2 - 6*s + 1)*s/((2*s - 3)^2*(s - 1))',
    'N_4^(8,3)': '(2*s - 1)*s/((2*s - 3)*(s - 1))',
    'N_5^(8,3)': '1/2*(2*s - 3)*s/(s - 2)^2',
    'N_6^(8,3)': '4*(s - 1)*s/(2*s - 3)^2',
    'N_7^(8,3)': '(6*s - 7)*s/((3*s - 5)*(2*s - 3))', # wt 1
    'N_8^(8,3)': 's/(s - 2)',
    'N_9^(8,3)': '2*s/(2*s - 3)', # wt 1
    'N_10^(8,3)': '(2*s + 1)*s/((2*s - 3)*(s - 1))',
    'N_11^(8,3)': '2*s^2/((2*s - 3)*(s - 1))',
    'N_1^(8,4)': '(s - 1)*s/((s - 2)*(s - 3))', # wt 1
    'N_2^(8,4)': '(s - 2)*s/(s - 3)^2',
    'N_3^(8,4)': '2*(s - 2)*s/((2*s - 5)*(s - 3))', # wt 1
    ####################################
    
    ####################################
    'N_3^(8,2)[eps]': '1/3*(96*s^3 - 204*s^2 + 136*s - 27)*s/((8*s - 7)*(4*s - 3)*(s - 1)^2)', # wt 2
    'N_4^(8,2)[eps]': '36*(s - 1)^2*s/((6*s - 7)*(3*s - 4)*(2*s - 3))', # wt 2

    'n(5,ZZ)': '1/2*(12*s^4 - 42*s^3 + 52*s^2 - 28*s + 7)*s^2/((3*s - 1)*(2*s - 3)*(s - 1)^2*(s - 2)^2)', # wt 1, dim 10
    'h(ZZ[X]/X^3)': '2*(3*s - 4)*s/((3*s - 5)*(2*s - 3))', # wt 1, dim 9
    'LL72_L1': '(36*s^3 - 141*s^2 + 167*s - 54)*(s - 1)^2*s/((4*s - 7)*(3*s - 4)*(3*s - 5)*(s - 2)^3)', # wt 1, dim 9
    'LL72_L3': '(s - 1)*s/(s - 2)^2', # dim 8

    'M3 curlyvee M3': '2*s^3/((2*s - 1)*(s - 1)^2)',
    ####################################
    }

# names[i] is a list of names attached to algebras[i]
# id[name] == i iff name in names[i]
# If known, topzetas[i]['subalgebras'] and topzetas[i]['ideals'] are the
# topological zeta functions of algebras[i] indicated.
# Later on, topzetas[i]['reps'] might be added too.

algebras = []
names = []
topzetas = []
id = {}

for coll, op in [ (_raw, lambda z: Algebra(**z.input)),
                  (_raw_tensored_with_duals, lambda z: tensor_with_duals(algebras[id[z.input]])) ]:
    for z in coll:
        assert len(algebras)==len(names)==len(topzetas)

        i = len(algebras)
        algebras.append(op(z))

        D = {}
        s = SR.var('s')
        if z.subalgebra_topzeta is not None:
            D['subalgebras'] = eval(preparse(z.subalgebra_topzeta))

        if z.ideal_topzeta is not None:
            D['ideals'] = eval(preparse(z.ideal_topzeta))
        topzetas.append(D)

        names.append(z.names)
        for name in z.names:
            id[name] = i

for name, z in _reps.items():
    i = id.get(name, None)
    if i is None:
        raise RuntimeError
    if z is not None:
        topzetas[i]['reps'] = eval(preparse(z))

padzetas = [{} for L in algebras]

for objs, D in _padic_raw.items():
    for name, zeta in D.items():
        try:
            i = id[name]
        except KeyError:
            raise RuntimeError('invalid name used in p-adic database')
        try:
            padzetas[i][str(objs)] = SR(str(zeta))
        except Exception:
            raise RuntimeError('possibly invalid entry in p-adic database')

def _essval_at_infinity(Z):
    R = QQ['s']
    s = R.gens()[0]
    K = FractionField(R)
    return (K(Z).numerator().leading_coefficient()/
            K(Z).denominator().leading_coefficient())

def _essval_at_zero(Z):
    return _essval_at_infinity(Z(s=1/s))

def _complex2str(z):
    a = z.real()
    b = z.imag()
    s = ''
    if a:
        s += '%.4f' % a.real()
        if b > 0:
            s += ' + '
        elif b < 0:
            s += ' - '
            b = -b
    if b:
        s += '%.4f*i' % b.real()
    return s if s else '0.0000'

def _order(Z,a):
    R = QQ['s']
    s = R.gen(0)
    return R(Z.numerator()).valuation(s-a) - R(Z.denominator()).valuation(s-a)
    
def _topzeta_invariants(Z, objects=None):
    PREC = 8192

    if not Z:
        return {'degree': -Infinity, 'order@zero': +Infinity,
                "ess. value@zero": 0, "ess. value@infinity": 0, 'zeta': '0' }
    if Z == 1:
        return {'degree': 0, 'order@zero': 0, 'ess. value@zero': 1, 'zeta': '1'}

    s = var('s')
    roots = [a for a,_ in Z.numerator(s).roots(s,ring=ComplexField(PREC))]
    roots.sort(key=lambda z:z.real())

    poles = [a for a,_ in Z.denominator(s).roots(s,ring=RationalField())] # ComplexField(PREC))]
    poles.sort(key=lambda z:z.real())

    D = {
        'zeta': str(Z),
        'degree': Z.degree(var('s')),
        'order@zero':_order(Z,0),
        'ess. value@zero': _essval_at_zero(Z),
        'ess. value@infinity': _essval_at_infinity(Z),
        'complex roots': ', '.join([_complex2str(a) for a in roots]) if roots else 'None',
        'poles': ', '.join([_complex2str(a) for a in poles]) if poles else 'None',
        '_roots': roots,
        '_poles': poles,
        }

    if objects == 'reps':
        Zm1 = (Z-1).factor()
        D['(zeta-1)@infinity'] = _essval_at_infinity(Zm1)
        D['degree(zeta-1)'] = Zm1.degree(s)

        f = (Z-s).factor().numerator(s)
        fixed_points = [a for a,_ in f.roots(s,ring=ComplexField(PREC))]
        fixed_points.sort(key=lambda z:z.real())

        D['fixed points'] = ', '.join([_complex2str(a) for a in fixed_points]) if fixed_points else 'None'
        D['_fixed_points'] = fixed_points

        # g = (Z-1).factor().numerator(s)
        # roots_zetam1 = [a for a,_ in g.roots(s,ring=ComplexField(PREC))]
        # roots_zetam1.sort(key=lambda z:z.real())

        # D['roots(zeta-1)'] = ', '.join([_complex2str(a) for a in roots_zetam1]) if roots_zetam1 else 'None'
        # D['_roots(zeta-1)'] = roots_zetam1
    return D
    
def _algebra_invariants(L):
    D = {
        'rank': L.rank,
        'Lie': L.is_Lie(),
        'commutative': L.is_commutative(),
        'associative': L.is_associative(),
        '_operators:': bool(L.operators),
        }

    if L.descr is not None:
        D['description'] = L.descr

    # I'll postpone considering nilpotency and solubility in the presence of
    # operators.
    if not L.operators: 
        D['soluble'] = L.is_soluble()
        D['nilpotent'] = L.is_nilpotent()

        if len(L.blocks) > 1:
            D['blocks'] = L.blocks[:]

        if L.is_soluble():
            D['derived length'] = L.derived_length()
        if L.is_nilpotent():
            D['nilpotency class'] = L.nilpotency_class()
            
        if L.is_Lie() and L.is_nilpotent():
            D['derived dim'] = L.derived_series()[1].rank()
            D['cocentral dim'] = L.rank - L._Lie_centre().rank()
            if len(L.blocks) == 1:
                R,S = L._SV_commutator_matrices()
                D['(u,v)'] = (R.rank()/Integer(2), S.rank())
    return D

def printall(file=None, subalgebras=True, ideals=True, reps=True):
    TAB = 8 * ' '
    def format_kv(key, value, width):
        return (('{:<%d}{}') % width).format(str(key) + ':', value)

    if file is None:
        import sys
        file = sys.stdout

    import common
    import datetime
    file.write('List of algebras and topological zeta functions.\n'
               'Created %s using Zeta %s.\n\n' %
               (str(datetime.datetime.now()), common.VERSION))


    pairs = []
    if subalgebras:
        pairs.append(('Topological subalgebra zeta function', 'subalgebras'))
    if ideals:
        pairs.append(('Topological ideal zeta function', 'ideals'))
    if reps:
        pairs.append(('Topological representation zeta function', 'reps'))
    cnt = -1

    for i, L in enumerate(algebras):
        if all(y not in topzetas[i] for x,y in pairs):
            continue

        cnt += 1
        file.write(72*'-' + '\n')
        file.write('ALGEBRA #%d\n' % cnt)
        file.write(72*'-' + '\n')
        file.write(format_kv('names', str(names[i])[1:-1], 18) + '\n',)

        file.write(format_kv('id', i, 18) + '\n')
        alginv = _algebra_invariants(L)
        for k,v in alginv.items():
            if k[0] != '_':
                file.write(format_kv(k, v, 18) + '\n')
        file.write('\n')

        for title, objects in pairs:
            Z = topzetas[i].get(objects, None)
            if Z is None:
                # file.write(TAB + 'not in list\n\n')
                continue

            file.write(title + ':\n')
            topinv = _topzeta_invariants(Z, objects)
            for k,v in topinv.items():
                if k[0] != '_':
                    file.write(TAB + format_kv(k, v, 22) + '\n')

            # Test some conjectures and known properties.
            if objects in ['subalgebras', 'ideals']:
                if topinv['zeta'] == '0' and alginv.get('nilpotent'):
                    file.write(TAB + format_kv('INTERESTING', 'a nilpotent algebra with trivial topological zeta function', 22) + '\n')

                if topinv['zeta'] != '0' and topinv['degree'] != -alginv['rank']:
                    file.write(TAB + format_kv('INTERESTING', 'counterexample to the degree conjecture', 22) + '\n')

                if topinv['zeta'] != '0' and topinv['order@zero'] >= 0:
                    file.write(TAB + format_kv('INTERESTING', 'zero is not a pole', 22) + '\n')

                if topinv.get('_roots'): # Hence, if topinv['zeta'] == '0', we do nothing.
                    if (topinv['_roots'][0].real() <= 0) or (topinv['_roots'][-1].real() >= alginv['rank'] - 2):
                        file.write(TAB + format_kv('INTERESTING', 'counterexample to the conjecture on the location of roots', 22) + '\n')

                if alginv.get('nilpotent') and not alginv.get('_operators') and len(L.blocks) == 1:
                    if topinv['order@zero'] != -1:
                        file.write(TAB + format_kv('INTERESTING', 'a nilpotent example without a simple pole at zero', 22) + '\n')
                    d = alginv['rank']
                    if topinv['ess. value@zero'] != Integer(-1)**(d-1)/Integer(factorial(d-1)):
                        file.write(TAB + format_kv('INTERESTING', 'counterexample to the conjecture on residues at zero', 22) + '\n')

            if objects == 'reps' and topinv['zeta'] != '1':
                if topinv['degree'] != 0:
                    file.write(TAB + format_kv('IMPOSSIBLE', 'degree is not zero', 22) + '\n')

                if topinv['order@zero'] <= 0:
                    file.write(TAB + format_kv('INTERESTING', 'does not vanish at zero', 22) + '\n')

                s = var('s')
                if (eval(preparse(topinv['zeta']))-1).factor().degree(s) != -1:
                    file.write(TAB + format_kv('INTERESTING', 'degree of zeta-1 is not -1', 22) + '\n')

                if topinv.get('_poles') and topinv.get('derived dim') and topinv['_poles'][-1] > topinv['derived dim']:
                    file.write(TAB + format_kv('IMPOSSIBLE', 'pole is too large', 22) + '\n')

                if topinv.get('(zeta-1)@infinity') and topinv['(zeta-1)@infinity'] <= 0:
                    file.write(TAB + format_kv('INTERESTING', 'essential value at infinity is non-positive', 22) + '\n')

                if topinv.get('_roots') and topinv['_roots'][0].imag():
                    file.write(TAB + format_kv('INTERESTING', 'smallest zero is not real', 22) + '\n')

                if topinv.get('_fixed_points') and topinv['_fixed_points'][0].imag():
                    file.write(TAB + format_kv('INTERESTING', 'smallest fixed point is not real', 22) + '\n')

                if topinv.get('_fixed_points') and topinv['_fixed_points'][-1].imag():
                    file.write(TAB + format_kv('INTERESTING', 'largest fixed point is not real', 22) + '\n')

                if topinv.get('_fixed_points') and topinv['_fixed_points'][-1].real() > alginv['rank'] - 1:
                    file.write(TAB + format_kv('INTERESTING', 'largest fixed point exceeds dim -1 ', 22) + '\n')

                if topinv.get('_roots') and topinv.get('_poles') and topinv['_roots'][-1].real() >= topinv['_poles'][-1]:
                    file.write(TAB + format_kv('INTERESTING', 'some zero exceeds the largest pole', 22) + '\n')

                if topinv.get('_poles') and topinv.get('_fixed_points') and topinv['_poles'][-1] >= topinv['_fixed_points'][-1].real():
                    file.write(TAB + format_kv('INTERESTING', 'some pole exceeds the largest fixed point', 22) + '\n')

            file.write('\n')
            file.flush()
